/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.CodecUtils;
import com.datastax.driver.core.DataType;
import com.datastax.driver.core.LocalDate;
import com.datastax.driver.core.ParseUtils;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.TupleType;
import com.datastax.driver.core.TupleValue;
import com.datastax.driver.core.UDTValue;
import com.datastax.driver.core.UserType;
import com.datastax.driver.core.exceptions.InvalidTypeException;
import com.datastax.driver.core.utils.Bytes;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.reflect.TypeToken;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

public abstract class TypeCodec<T> {
    private static final Map<TypeToken<?>, TypeToken<?>> primitiveToWrapperMap = ImmutableMap.builder().put((Object)TypeToken.of(Boolean.TYPE), (Object)TypeToken.of(Boolean.class)).put((Object)TypeToken.of(Byte.TYPE), (Object)TypeToken.of(Byte.class)).put((Object)TypeToken.of(Character.TYPE), (Object)TypeToken.of(Character.class)).put((Object)TypeToken.of(Short.TYPE), (Object)TypeToken.of(Short.class)).put((Object)TypeToken.of(Integer.TYPE), (Object)TypeToken.of(Integer.class)).put((Object)TypeToken.of(Long.TYPE), (Object)TypeToken.of(Long.class)).put((Object)TypeToken.of(Double.TYPE), (Object)TypeToken.of(Double.class)).put((Object)TypeToken.of(Float.TYPE), (Object)TypeToken.of(Float.class)).build();
    protected final TypeToken<T> javaType;
    protected final DataType cqlType;

    public static PrimitiveBooleanCodec cboolean() {
        return BooleanCodec.instance;
    }

    public static PrimitiveByteCodec tinyInt() {
        return TinyIntCodec.instance;
    }

    public static PrimitiveShortCodec smallInt() {
        return SmallIntCodec.instance;
    }

    public static PrimitiveIntCodec cint() {
        return IntCodec.instance;
    }

    public static PrimitiveLongCodec bigint() {
        return BigintCodec.instance;
    }

    public static PrimitiveLongCodec counter() {
        return CounterCodec.instance;
    }

    public static PrimitiveFloatCodec cfloat() {
        return FloatCodec.instance;
    }

    public static PrimitiveDoubleCodec cdouble() {
        return DoubleCodec.instance;
    }

    public static TypeCodec<BigInteger> varint() {
        return VarintCodec.instance;
    }

    public static TypeCodec<BigDecimal> decimal() {
        return DecimalCodec.instance;
    }

    public static TypeCodec<String> ascii() {
        return AsciiCodec.instance;
    }

    public static TypeCodec<String> varchar() {
        return VarcharCodec.instance;
    }

    public static TypeCodec<ByteBuffer> blob() {
        return BlobCodec.instance;
    }

    public static TypeCodec<LocalDate> date() {
        return DateCodec.instance;
    }

    public static TypeCodec<Long> time() {
        return TimeCodec.instance;
    }

    public static TypeCodec<Date> timestamp() {
        return TimestampCodec.instance;
    }

    public static TypeCodec<UUID> uuid() {
        return UUIDCodec.instance;
    }

    public static TypeCodec<UUID> timeUUID() {
        return TimeUUIDCodec.instance;
    }

    public static TypeCodec<InetAddress> inet() {
        return InetCodec.instance;
    }

    public static <T> TypeCodec<List<T>> list(TypeCodec<T> elementCodec) {
        return new ListCodec(elementCodec);
    }

    public static <T> TypeCodec<Set<T>> set(TypeCodec<T> elementCodec) {
        return new SetCodec(elementCodec);
    }

    public static <K, V> TypeCodec<Map<K, V>> map(TypeCodec<K> keyCodec, TypeCodec<V> valueCodec) {
        return new MapCodec(keyCodec, valueCodec);
    }

    public static TypeCodec<UDTValue> userType(UserType type) {
        return new UDTCodec(type);
    }

    public static TypeCodec<TupleValue> tuple(TupleType type) {
        return new TupleCodec(type);
    }

    public static TypeCodec<ByteBuffer> custom(DataType.CustomType type) {
        return new CustomCodec((DataType)type);
    }

    protected TypeCodec(DataType cqlType, Class<T> javaClass) {
        this(cqlType, TypeToken.of(javaClass));
    }

    protected TypeCodec(DataType cqlType, TypeToken<T> javaType) {
        Preconditions.checkNotNull((Object)cqlType, (Object)"cqlType cannot be null");
        Preconditions.checkNotNull(javaType, (Object)"javaType cannot be null");
        Preconditions.checkArgument((!javaType.isPrimitive() ? 1 : 0) != 0, (String)"Cannot create a codec for a primitive Java type (%s), please use the wrapper type instead", (Object[])new Object[]{javaType});
        this.cqlType = cqlType;
        this.javaType = javaType;
    }

    public TypeToken<T> getJavaType() {
        return this.javaType;
    }

    public DataType getCqlType() {
        return this.cqlType;
    }

    public abstract ByteBuffer serialize(T var1, ProtocolVersion var2) throws InvalidTypeException;

    public abstract T deserialize(ByteBuffer var1, ProtocolVersion var2) throws InvalidTypeException;

    public abstract T parse(String var1) throws InvalidTypeException;

    public abstract String format(T var1) throws InvalidTypeException;

    public boolean accepts(TypeToken javaType) {
        Preconditions.checkNotNull((Object)javaType);
        if (javaType.isPrimitive()) {
            javaType = primitiveToWrapperMap.get(javaType);
        }
        return this.javaType.isAssignableFrom((TypeToken)javaType);
    }

    public boolean accepts(DataType cqlType) {
        Preconditions.checkNotNull((Object)cqlType);
        return this.cqlType.equals(cqlType);
    }

    public boolean accepts(Object value) {
        Preconditions.checkNotNull((Object)value);
        return this.accepts(TypeToken.of(value.getClass()));
    }

    public String toString() {
        return String.format("%s [%s <-> %s]", this.getClass().getSimpleName(), this.cqlType, this.javaType);
    }

    private static class TupleCodec
    extends TypeCodec<TupleValue> {
        private final TupleType definition;

        private TupleCodec(TupleType definition) {
            super((DataType)definition, TupleValue.class);
            this.definition = definition;
        }

        @Override
        public boolean accepts(DataType cqlType) {
            return cqlType instanceof TupleType && this.definition.contains((TupleType)cqlType);
        }

        @Override
        public boolean accepts(Object value) {
            return value instanceof TupleValue && this.definition.contains(((TupleValue)value).getType());
        }

        @Override
        public TupleValue parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            TupleValue v = this.definition.newValue();
            int idx = ParseUtils.skipSpaces(value, 0);
            if (value.charAt(idx++) != '(') {
                throw new InvalidTypeException(String.format("Cannot parse tuple value from \"%s\", at character %d expecting '(' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
            }
            if (value.charAt(idx = ParseUtils.skipSpaces(value, idx)) == ')') {
                return v;
            }
            int i = 0;
            while (idx < value.length()) {
                int n;
                try {
                    n = ParseUtils.skipCQLValue(value, idx);
                }
                catch (IllegalArgumentException e) {
                    throw new InvalidTypeException(String.format("Cannot parse tuple value from \"%s\", invalid CQL value at character %d", value, idx), e);
                }
                DataType dt = this.definition.getComponentTypes().get(i);
                TypeCodec codec = this.definition.getCodecRegistry().codecFor(dt);
                v.set(i, codec.parse(value.substring(idx, n)), codec.getJavaType());
                idx = n;
                ++i;
                idx = ParseUtils.skipSpaces(value, idx);
                if (value.charAt(idx) == ')') {
                    return v;
                }
                if (value.charAt(idx) != ',') {
                    throw new InvalidTypeException(String.format("Cannot parse tuple value from \"%s\", at character %d expecting ',' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
                }
                ++idx;
                idx = ParseUtils.skipSpaces(value, idx);
            }
            throw new InvalidTypeException(String.format("Malformed tuple value \"%s\", missing closing ')'", value));
        }

        @Override
        public String format(TupleValue value) {
            if (value == null) {
                return "NULL";
            }
            return value.toString();
        }

        @Override
        public ByteBuffer serialize(TupleValue value, ProtocolVersion protocolVersion) {
            if (value == null) {
                return null;
            }
            int size = 0;
            for (ByteBuffer v : value.values) {
                size += 4 + (v == null ? 0 : v.remaining());
            }
            ByteBuffer result = ByteBuffer.allocate(size);
            for (ByteBuffer bb : value.values) {
                if (bb == null) {
                    result.putInt(-1);
                    continue;
                }
                result.putInt(bb.remaining());
                result.put(bb.duplicate());
            }
            return (ByteBuffer)result.flip();
        }

        @Override
        public TupleValue deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null) {
                return null;
            }
            ByteBuffer input = bytes.duplicate();
            TupleValue value = this.definition.newValue();
            int i = 0;
            while (input.hasRemaining() && i < value.values.length) {
                int n = input.getInt();
                value.values[i++] = n < 0 ? null : CodecUtils.readBytes(input, n);
            }
            return value;
        }
    }

    private static class UDTCodec
    extends TypeCodec<UDTValue> {
        private final UserType definition;

        private UDTCodec(UserType definition) {
            super((DataType)definition, UDTValue.class);
            this.definition = definition;
        }

        @Override
        public boolean accepts(Object value) {
            return value instanceof UDTValue && ((UDTValue)value).getType().equals(this.definition);
        }

        @Override
        public UDTValue parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            UDTValue v = this.definition.newValue();
            int idx = ParseUtils.skipSpaces(value, 0);
            if (value.charAt(idx++) != '{') {
                throw new InvalidTypeException(String.format("Cannot parse UDT value from \"%s\", at character %d expecting '{' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
            }
            if (value.charAt(idx = ParseUtils.skipSpaces(value, idx)) == '}') {
                return v;
            }
            while (idx < value.length()) {
                int n;
                try {
                    n = ParseUtils.skipCQLId(value, idx);
                }
                catch (IllegalArgumentException e) {
                    throw new InvalidTypeException(String.format("Cannot parse UDT value from \"%s\", cannot parse a CQL identifier at character %d", value, idx), e);
                }
                String name = value.substring(idx, n);
                idx = n;
                if (!this.definition.contains(name)) {
                    throw new InvalidTypeException(String.format("Unknown field %s in value \"%s\"", name, value));
                }
                idx = ParseUtils.skipSpaces(value, idx);
                if (value.charAt(idx++) != ':') {
                    throw new InvalidTypeException(String.format("Cannot parse UDT value from \"%s\", at character %d expecting ':' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
                }
                idx = ParseUtils.skipSpaces(value, idx);
                try {
                    n = ParseUtils.skipCQLValue(value, idx);
                }
                catch (IllegalArgumentException e) {
                    throw new InvalidTypeException(String.format("Cannot parse UDT value from \"%s\", invalid CQL value at character %d", value, idx), e);
                }
                DataType dt = this.definition.getFieldType(name);
                TypeCodec codec = this.definition.getCodecRegistry().codecFor(dt);
                v.set(name, codec.parse(value.substring(idx, n)), codec.getJavaType());
                idx = n;
                idx = ParseUtils.skipSpaces(value, idx);
                if (value.charAt(idx) == '}') {
                    return v;
                }
                if (value.charAt(idx) != ',') {
                    throw new InvalidTypeException(String.format("Cannot parse UDT value from \"%s\", at character %d expecting ',' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
                }
                ++idx;
                idx = ParseUtils.skipSpaces(value, idx);
            }
            throw new InvalidTypeException(String.format("Malformed UDT value \"%s\", missing closing '}'", value));
        }

        @Override
        public String format(UDTValue value) {
            if (value == null) {
                return "NULL";
            }
            return value.toString();
        }

        @Override
        public ByteBuffer serialize(UDTValue value, ProtocolVersion protocolVersion) {
            if (value == null) {
                return null;
            }
            int size = 0;
            for (ByteBuffer v : value.values) {
                size += 4 + (v == null ? 0 : v.remaining());
            }
            ByteBuffer result = ByteBuffer.allocate(size);
            for (ByteBuffer bb : value.values) {
                if (bb == null) {
                    result.putInt(-1);
                    continue;
                }
                result.putInt(bb.remaining());
                result.put(bb.duplicate());
            }
            return (ByteBuffer)result.flip();
        }

        @Override
        public UDTValue deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null) {
                return null;
            }
            ByteBuffer input = bytes.duplicate();
            UDTValue value = this.definition.newValue();
            int i = 0;
            while (input.hasRemaining() && i < value.values.length) {
                int n = input.getInt();
                value.values[i++] = n < 0 ? null : CodecUtils.readBytes(input, n);
            }
            return value;
        }
    }

    private static class MapCodec<K, V>
    extends TypeCodec<Map<K, V>> {
        private final TypeCodec<K> keyCodec;
        private final TypeCodec<V> valueCodec;

        private MapCodec(TypeCodec<K> keyCodec, TypeCodec<V> valueCodec) {
            super((DataType)DataType.map(keyCodec.getCqlType(), valueCodec.getCqlType()), CodecUtils.mapOf(keyCodec.getJavaType(), valueCodec.getJavaType()));
            this.keyCodec = keyCodec;
            this.valueCodec = valueCodec;
        }

        @Override
        public boolean accepts(Object value) {
            if (value instanceof Map) {
                Map map = (Map)value;
                if (map.isEmpty()) {
                    return true;
                }
                Map.Entry entry = map.entrySet().iterator().next();
                return this.keyCodec.accepts(entry.getKey()) && this.valueCodec.accepts(entry.getValue());
            }
            return false;
        }

        @Override
        public Map<K, V> parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            int idx = ParseUtils.skipSpaces(value, 0);
            if (value.charAt(idx++) != '{') {
                throw new InvalidTypeException(String.format("cannot parse map value from \"%s\", at character %d expecting '{' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
            }
            if (value.charAt(idx = ParseUtils.skipSpaces(value, idx)) == '}') {
                return new LinkedHashMap(0);
            }
            HashMap<K, V> m = new HashMap<K, V>();
            while (idx < value.length()) {
                int n;
                try {
                    n = ParseUtils.skipCQLValue(value, idx);
                }
                catch (IllegalArgumentException e) {
                    throw new InvalidTypeException(String.format("Cannot parse map value from \"%s\", invalid CQL value at character %d", value, idx), e);
                }
                K k = this.keyCodec.parse(value.substring(idx, n));
                idx = n;
                idx = ParseUtils.skipSpaces(value, idx);
                if (value.charAt(idx++) != ':') {
                    throw new InvalidTypeException(String.format("Cannot parse map value from \"%s\", at character %d expecting ':' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
                }
                idx = ParseUtils.skipSpaces(value, idx);
                try {
                    n = ParseUtils.skipCQLValue(value, idx);
                }
                catch (IllegalArgumentException e) {
                    throw new InvalidTypeException(String.format("Cannot parse map value from \"%s\", invalid CQL value at character %d", value, idx), e);
                }
                V v = this.valueCodec.parse(value.substring(idx, n));
                idx = n;
                m.put(k, v);
                idx = ParseUtils.skipSpaces(value, idx);
                if (value.charAt(idx) == '}') {
                    return m;
                }
                if (value.charAt(idx++) != ',') {
                    throw new InvalidTypeException(String.format("Cannot parse map value from \"%s\", at character %d expecting ',' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
                }
                idx = ParseUtils.skipSpaces(value, idx);
            }
            throw new InvalidTypeException(String.format("Malformed map value \"%s\", missing closing '}'", value));
        }

        @Override
        public String format(Map<K, V> value) {
            if (value == null) {
                return "NULL";
            }
            StringBuilder sb = new StringBuilder();
            sb.append("{");
            int i = 0;
            for (Map.Entry<K, V> e : value.entrySet()) {
                if (i++ != 0) {
                    sb.append(",");
                }
                sb.append(this.keyCodec.format(e.getKey()));
                sb.append(":");
                sb.append(this.valueCodec.format(e.getValue()));
            }
            sb.append("}");
            return sb.toString();
        }

        @Override
        public ByteBuffer serialize(Map<K, V> value, ProtocolVersion protocolVersion) {
            if (value == null) {
                return null;
            }
            ArrayList<ByteBuffer> bbs = new ArrayList<ByteBuffer>(2 * value.size());
            for (Map.Entry<K, V> entry : value.entrySet()) {
                ByteBuffer bbv;
                ByteBuffer bbk;
                K key = entry.getKey();
                if (key == null) {
                    throw new NullPointerException("Map keys cannot be null");
                }
                try {
                    bbk = this.keyCodec.serialize(key, protocolVersion);
                }
                catch (ClassCastException e) {
                    throw new InvalidTypeException(String.format("Invalid type for map key, expecting % but got %s", this.keyCodec.getJavaType(), key.getClass()), e);
                }
                V v = entry.getValue();
                if (v == null) {
                    throw new NullPointerException("Map values cannot be null");
                }
                try {
                    bbv = this.valueCodec.serialize(v, protocolVersion);
                }
                catch (ClassCastException e) {
                    throw new InvalidTypeException(String.format("Invalid type for map value, expecting % but got %s", this.valueCodec.getJavaType(), v.getClass()), e);
                }
                bbs.add(bbk);
                bbs.add(bbv);
            }
            return CodecUtils.pack(bbs, value.size(), protocolVersion);
        }

        @Override
        public Map<K, V> deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return new LinkedHashMap(0);
            }
            try {
                ByteBuffer input = bytes.duplicate();
                int n = CodecUtils.readCollectionSize(input, protocolVersion);
                LinkedHashMap<K, V> m = new LinkedHashMap<K, V>(n);
                for (int i = 0; i < n; ++i) {
                    ByteBuffer kbb = CodecUtils.readCollectionValue(input, protocolVersion);
                    ByteBuffer vbb = CodecUtils.readCollectionValue(input, protocolVersion);
                    m.put(this.keyCodec.deserialize(kbb, protocolVersion), this.valueCodec.deserialize(vbb, protocolVersion));
                }
                return m;
            }
            catch (BufferUnderflowException e) {
                throw new InvalidTypeException("Not enough bytes to deserialize a map");
            }
        }
    }

    private static class SetCodec<T>
    extends CollectionCodec<T, Set<T>> {
        private SetCodec(TypeCodec<T> eltCodec) {
            super(DataType.set(eltCodec.cqlType), CodecUtils.setOf(eltCodec.getJavaType()), eltCodec);
        }

        @Override
        protected Set<T> newInstance(int capacity) {
            return new LinkedHashSet(capacity);
        }

        @Override
        protected char getOpeningChar() {
            return '{';
        }

        @Override
        protected char getClosingChar() {
            return '}';
        }
    }

    private static class ListCodec<T>
    extends CollectionCodec<T, List<T>> {
        private ListCodec(TypeCodec<T> eltCodec) {
            super(DataType.list(eltCodec.getCqlType()), CodecUtils.listOf(eltCodec.getJavaType()), eltCodec);
        }

        @Override
        protected List<T> newInstance(int capacity) {
            return new ArrayList(capacity);
        }

        @Override
        protected char getOpeningChar() {
            return '[';
        }

        @Override
        protected char getClosingChar() {
            return ']';
        }
    }

    private static abstract class CollectionCodec<E, C extends Collection<E>>
    extends TypeCodec<C> {
        protected final TypeCodec<E> eltCodec;

        private CollectionCodec(DataType.CollectionType cqlType, TypeToken<C> javaType, TypeCodec<E> eltCodec) {
            super((DataType)cqlType, javaType);
            this.eltCodec = eltCodec;
        }

        @Override
        public ByteBuffer serialize(C value, ProtocolVersion protocolVersion) {
            if (value == null) {
                return null;
            }
            ArrayList<ByteBuffer> bbs = new ArrayList<ByteBuffer>(value.size());
            for (Object elt : value) {
                ByteBuffer bb;
                if (elt == null) {
                    throw new NullPointerException("Collection elements cannot be null");
                }
                try {
                    bb = this.eltCodec.serialize(elt, protocolVersion);
                }
                catch (ClassCastException e) {
                    throw new InvalidTypeException(String.format("Invalid type for %s element, expecting %s but got %s", this.cqlType, this.eltCodec.getJavaType(), elt.getClass()), e);
                }
                bbs.add(bb);
            }
            return CodecUtils.pack(bbs, value.size(), protocolVersion);
        }

        @Override
        public C deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return this.newInstance(0);
            }
            try {
                ByteBuffer input = bytes.duplicate();
                int n = CodecUtils.readCollectionSize(input, protocolVersion);
                C coll = this.newInstance(n);
                for (int i = 0; i < n; ++i) {
                    ByteBuffer databb = CodecUtils.readCollectionValue(input, protocolVersion);
                    coll.add(this.eltCodec.deserialize(databb, protocolVersion));
                }
                return coll;
            }
            catch (BufferUnderflowException e) {
                throw new InvalidTypeException("Not enough bytes to deserialize list");
            }
        }

        @Override
        public String format(C value) {
            if (value == null) {
                return "NULL";
            }
            StringBuilder sb = new StringBuilder();
            sb.append(this.getOpeningChar());
            int i = 0;
            for (Object v : value) {
                if (i++ != 0) {
                    sb.append(",");
                }
                sb.append(this.eltCodec.format(v));
            }
            sb.append(this.getClosingChar());
            return sb.toString();
        }

        @Override
        public C parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            int idx = ParseUtils.skipSpaces(value, 0);
            if (value.charAt(idx++) != this.getOpeningChar()) {
                throw new InvalidTypeException(String.format("cannot parse list value from \"%s\", at character %d expecting '[' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
            }
            if (value.charAt(idx = ParseUtils.skipSpaces(value, idx)) == this.getClosingChar()) {
                return this.newInstance(0);
            }
            C l = this.newInstance(10);
            while (idx < value.length()) {
                int n;
                try {
                    n = ParseUtils.skipCQLValue(value, idx);
                }
                catch (IllegalArgumentException e) {
                    throw new InvalidTypeException(String.format("Cannot parse list value from \"%s\", invalid CQL value at character %d", value, idx), e);
                }
                l.add(this.eltCodec.parse(value.substring(idx, n)));
                idx = n;
                idx = ParseUtils.skipSpaces(value, idx);
                if (value.charAt(idx) == this.getClosingChar()) {
                    return l;
                }
                if (value.charAt(idx++) != ',') {
                    throw new InvalidTypeException(String.format("Cannot parse list value from \"%s\", at character %d expecting ',' but got '%c'", value, idx, Character.valueOf(value.charAt(idx))));
                }
                idx = ParseUtils.skipSpaces(value, idx);
            }
            throw new InvalidTypeException(String.format("Malformed list value \"%s\", missing closing ']'", value));
        }

        @Override
        public boolean accepts(Object value) {
            if (this.getJavaType().getRawType().isAssignableFrom(value.getClass())) {
                Collection list = (Collection)value;
                if (list.isEmpty()) {
                    return true;
                }
                Object elt = list.iterator().next();
                return this.eltCodec.accepts(elt);
            }
            return false;
        }

        protected abstract C newInstance(int var1);

        protected abstract char getOpeningChar();

        protected abstract char getClosingChar();
    }

    private static class VarintCodec
    extends TypeCodec<BigInteger> {
        private static final VarintCodec instance = new VarintCodec();

        private VarintCodec() {
            super(DataType.varint(), BigInteger.class);
        }

        @Override
        public BigInteger parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : new BigInteger(value);
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse varint value from \"%s\"", value));
            }
        }

        @Override
        public String format(BigInteger value) {
            if (value == null) {
                return "NULL";
            }
            return value.toString();
        }

        @Override
        public ByteBuffer serialize(BigInteger value, ProtocolVersion protocolVersion) {
            return value == null ? null : ByteBuffer.wrap(value.toByteArray());
        }

        @Override
        public BigInteger deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : new BigInteger(Bytes.getArray(bytes));
        }
    }

    private static class TimeUUIDCodec
    extends AbstractUUIDCodec {
        private static final TimeUUIDCodec instance = new TimeUUIDCodec();

        private TimeUUIDCodec() {
            super(DataType.timeuuid());
        }

        @Override
        public String format(UUID value) {
            if (value == null) {
                return "NULL";
            }
            if (value.version() != 1) {
                throw new InvalidTypeException(String.format("%s is not a Type 1 (time-based) UUID", value));
            }
            return super.format(value);
        }

        @Override
        public ByteBuffer serialize(UUID value, ProtocolVersion protocolVersion) {
            if (value == null) {
                return null;
            }
            if (value.version() != 1) {
                throw new InvalidTypeException(String.format("%s is not a Type 1 (time-based) UUID", value));
            }
            return super.serialize(value, protocolVersion);
        }
    }

    private static class UUIDCodec
    extends AbstractUUIDCodec {
        private static final UUIDCodec instance = new UUIDCodec();

        private UUIDCodec() {
            super(DataType.uuid());
        }
    }

    private static abstract class AbstractUUIDCodec
    extends TypeCodec<UUID> {
        private AbstractUUIDCodec(DataType cqlType) {
            super(cqlType, UUID.class);
        }

        @Override
        public UUID parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : UUID.fromString(value);
            }
            catch (IllegalArgumentException e) {
                throw new InvalidTypeException(String.format("Cannot parse UUID value from \"%s\"", value));
            }
        }

        @Override
        public String format(UUID value) {
            if (value == null) {
                return "NULL";
            }
            return value.toString();
        }

        @Override
        public ByteBuffer serialize(UUID value, ProtocolVersion protocolVersion) {
            if (value == null) {
                return null;
            }
            ByteBuffer bb = ByteBuffer.allocate(16);
            bb.putLong(0, value.getMostSignificantBits());
            bb.putLong(8, value.getLeastSignificantBits());
            return bb;
        }

        @Override
        public UUID deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : new UUID(bytes.getLong(bytes.position()), bytes.getLong(bytes.position() + 8));
        }
    }

    private static class TimeCodec
    extends LongCodec {
        private static final Pattern IS_LONG_PATTERN = Pattern.compile("^-?\\d+$");
        private static final TimeCodec instance = new TimeCodec();

        private TimeCodec() {
            super(DataType.time());
        }

        private static Long parseTime(String s) throws IllegalArgumentException {
            long second;
            long minute;
            long hour;
            long a_nanos = 0L;
            String formatError = "Timestamp format must be hh:mm:ss[.fffffffff]";
            String zeros = "000000000";
            if (s == null) {
                throw new IllegalArgumentException(formatError);
            }
            s = s.trim();
            int firstColon = s.indexOf(58);
            int secondColon = s.indexOf(58, firstColon + 1);
            if (firstColon > 0 && secondColon > 0 && secondColon < s.length() - 1) {
                int period = s.indexOf(46, secondColon + 1);
                hour = Integer.parseInt(s.substring(0, firstColon));
                if (hour < 0L || hour >= 24L) {
                    throw new IllegalArgumentException("Hour out of bounds.");
                }
                minute = Integer.parseInt(s.substring(firstColon + 1, secondColon));
                if (minute < 0L || minute >= 60L) {
                    throw new IllegalArgumentException("Minute out of bounds.");
                }
                if (period > 0 && period < s.length() - 1) {
                    second = Integer.parseInt(s.substring(secondColon + 1, period));
                    if (second < 0L || second >= 60L) {
                        throw new IllegalArgumentException("Second out of bounds.");
                    }
                    String nanos_s = s.substring(period + 1);
                    if (nanos_s.length() > 9) {
                        throw new IllegalArgumentException(formatError);
                    }
                    if (!Character.isDigit(nanos_s.charAt(0))) {
                        throw new IllegalArgumentException(formatError);
                    }
                    nanos_s = nanos_s + zeros.substring(0, 9 - nanos_s.length());
                    a_nanos = Integer.parseInt(nanos_s);
                } else {
                    if (period > 0) {
                        throw new IllegalArgumentException(formatError);
                    }
                    second = Integer.parseInt(s.substring(secondColon + 1));
                    if (second < 0L || second >= 60L) {
                        throw new IllegalArgumentException("Second out of bounds.");
                    }
                }
            } else {
                throw new IllegalArgumentException(formatError);
            }
            long rawTime = 0L;
            rawTime += TimeUnit.HOURS.toNanos(hour);
            rawTime += TimeUnit.MINUTES.toNanos(minute);
            rawTime += TimeUnit.SECONDS.toNanos(second);
            return rawTime += a_nanos;
        }

        @Override
        public Long parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            if (value.charAt(0) != '\'' || value.charAt(value.length() - 1) != '\'') {
                throw new InvalidTypeException("time values must be enclosed by single quotes");
            }
            if (IS_LONG_PATTERN.matcher(value = value.substring(1, value.length() - 1)).matches()) {
                try {
                    return Long.parseLong(value);
                }
                catch (NumberFormatException e) {
                    throw new InvalidTypeException(String.format("Cannot parse time value from \"%s\"", value));
                }
            }
            try {
                return TimeCodec.parseTime(value);
            }
            catch (IllegalArgumentException e) {
                throw new InvalidTypeException(String.format("Cannot parse time value from \"%s\"", value));
            }
        }

        @Override
        public String format(Long value) {
            if (value == null) {
                return "NULL";
            }
            int nano = (int)(value % 1000000000L);
            value = value - (long)nano;
            value = value / 1000000000L;
            int seconds = (int)(value % 60L);
            value = value - (long)seconds;
            value = value / 60L;
            int minutes = (int)(value % 60L);
            value = value - (long)minutes;
            value = value / 60L;
            int hours = (int)(value % 24L);
            value = value - (long)hours;
            value = value / 24L;
            assert (value == 0L);
            StringBuilder sb = new StringBuilder("'");
            TimeCodec.leftPadZeros(hours, 2, sb);
            sb.append(":");
            TimeCodec.leftPadZeros(minutes, 2, sb);
            sb.append(":");
            TimeCodec.leftPadZeros(seconds, 2, sb);
            sb.append(".");
            TimeCodec.leftPadZeros(nano, 9, sb);
            sb.append("'");
            return sb.toString();
        }

        private static void leftPadZeros(int value, int digits, StringBuilder sb) {
            sb.append(String.format("%0" + digits + "d", value));
        }
    }

    private static class DateCodec
    extends TypeCodec<LocalDate> {
        private static final DateCodec instance = new DateCodec();
        private static final Pattern IS_LONG_PATTERN = Pattern.compile("^-?\\d+$");
        private static final String pattern = "yyyy-MM-dd";
        private static final long MAX_LONG_VALUE = 0xFFFFFFFFL;
        private static final long EPOCH_AS_CQL_LONG = 0x80000000L;

        private DateCodec() {
            super(DataType.date(), LocalDate.class);
        }

        @Override
        public LocalDate parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            if (value.charAt(0) == '\'' && value.charAt(value.length() - 1) == '\'') {
                value = value.substring(1, value.length() - 1);
            }
            if (IS_LONG_PATTERN.matcher(value).matches()) {
                long cqlLong;
                try {
                    cqlLong = Long.parseLong(value);
                }
                catch (NumberFormatException e) {
                    throw new InvalidTypeException(String.format("Cannot parse date value from \"%s\"", value));
                }
                if (cqlLong < 0L || cqlLong > 0xFFFFFFFFL) {
                    throw new InvalidTypeException(String.format("Numeric literals for DATE must be between 0 and %d (got %d)", 0xFFFFFFFFL, cqlLong));
                }
                int days = (int)(cqlLong - 0x80000000L);
                return LocalDate.fromDaysSinceEpoch(days);
            }
            SimpleDateFormat parser = new SimpleDateFormat(pattern);
            parser.setLenient(false);
            parser.setTimeZone(TimeZone.getTimeZone("UTC"));
            ParsePosition pos = new ParsePosition(0);
            Date date = parser.parse(value, pos);
            if (date != null && pos.getIndex() == value.length()) {
                return LocalDate.fromMillisSinceEpoch(date.getTime());
            }
            throw new InvalidTypeException(String.format("Cannot parse date value from \"%s\"", value));
        }

        @Override
        public String format(LocalDate value) {
            if (value == null) {
                return "NULL";
            }
            return "'" + value.toString() + "'";
        }

        @Override
        public ByteBuffer serialize(LocalDate value, ProtocolVersion protocolVersion) {
            return value == null ? null : IntCodec.instance.serializeNoBoxing(DateCodec.javaToProtocol(value.getDaysSinceEpoch()), protocolVersion);
        }

        @Override
        public LocalDate deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : LocalDate.fromDaysSinceEpoch(DateCodec.protocolToJava(IntCodec.instance.deserializeNoBoxing(bytes, protocolVersion)));
        }

        private static int protocolToJava(int p) {
            return p + Integer.MIN_VALUE;
        }

        private static int javaToProtocol(int j) {
            return j - Integer.MIN_VALUE;
        }
    }

    private static class TimestampCodec
    extends TypeCodec<Date> {
        private static final String[] iso8601Patterns = new String[]{"yyyy-MM-dd HH:mm", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mmZ", "yyyy-MM-dd HH:mm:ssZ", "yyyy-MM-dd HH:mm:ss.SSS", "yyyy-MM-dd HH:mm:ss.SSSZ", "yyyy-MM-dd'T'HH:mm", "yyyy-MM-dd'T'HH:mmZ", "yyyy-MM-dd'T'HH:mm:ss", "yyyy-MM-dd'T'HH:mm:ssZ", "yyyy-MM-dd'T'HH:mm:ss.SSS", "yyyy-MM-dd'T'HH:mm:ss.SSSZ", "yyyy-MM-dd", "yyyy-MM-ddZ"};
        private static final TimestampCodec instance = new TimestampCodec();
        private static final Pattern IS_LONG_PATTERN = Pattern.compile("^-?\\d+$");

        private TimestampCodec() {
            super(DataType.timestamp(), Date.class);
        }

        private static Date parseDate(String str, String[] parsePatterns) throws ParseException {
            SimpleDateFormat parser = new SimpleDateFormat();
            parser.setLenient(false);
            ParsePosition pos = new ParsePosition(0);
            for (String parsePattern : parsePatterns) {
                parser.applyPattern(parsePattern);
                pos.setIndex(0);
                Date date = parser.parse(str, pos);
                if (date == null || pos.getIndex() != str.length()) continue;
                return date;
            }
            throw new ParseException("Unable to parse the date: " + str, -1);
        }

        @Override
        public Date parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            if (value.charAt(0) == '\'' && value.charAt(value.length() - 1) == '\'') {
                value = value.substring(1, value.length() - 1);
            }
            if (IS_LONG_PATTERN.matcher(value).matches()) {
                try {
                    return new Date(Long.parseLong(value));
                }
                catch (NumberFormatException e) {
                    throw new InvalidTypeException(String.format("Cannot parse timestamp value from \"%s\"", value));
                }
            }
            try {
                return TimestampCodec.parseDate(value, iso8601Patterns);
            }
            catch (ParseException e) {
                throw new InvalidTypeException(String.format("Cannot parse timestamp value from \"%s\"", value));
            }
        }

        @Override
        public String format(Date value) {
            if (value == null) {
                return "NULL";
            }
            return Long.toString(value.getTime());
        }

        @Override
        public ByteBuffer serialize(Date value, ProtocolVersion protocolVersion) {
            return value == null ? null : BigintCodec.instance.serializeNoBoxing(value.getTime(), protocolVersion);
        }

        @Override
        public Date deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : new Date(BigintCodec.instance.deserializeNoBoxing(bytes, protocolVersion));
        }
    }

    private static class IntCodec
    extends PrimitiveIntCodec {
        private static final IntCodec instance = new IntCodec();

        private IntCodec() {
            super(DataType.cint());
        }

        @Override
        public Integer parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : Integer.valueOf(Integer.parseInt(value));
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse 32-bits int value from \"%s\"", value));
            }
        }

        @Override
        public String format(Integer value) {
            if (value == null) {
                return "NULL";
            }
            return Integer.toString(value);
        }

        @Override
        public ByteBuffer serializeNoBoxing(int value, ProtocolVersion protocolVersion) {
            ByteBuffer bb = ByteBuffer.allocate(4);
            bb.putInt(0, value);
            return bb;
        }

        @Override
        public int deserializeNoBoxing(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return 0;
            }
            if (bytes.remaining() != 4) {
                throw new InvalidTypeException("Invalid 32-bits integer value, expecting 4 bytes but got " + bytes.remaining());
            }
            return bytes.getInt(bytes.position());
        }
    }

    private static class SmallIntCodec
    extends PrimitiveShortCodec {
        private static final SmallIntCodec instance = new SmallIntCodec();

        private SmallIntCodec() {
            super(DataType.smallint());
        }

        @Override
        public Short parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : Short.valueOf(Short.parseShort(value));
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse 16-bits int value from \"%s\"", value));
            }
        }

        @Override
        public String format(Short value) {
            if (value == null) {
                return "NULL";
            }
            return Short.toString(value);
        }

        @Override
        public ByteBuffer serializeNoBoxing(short value, ProtocolVersion protocolVersion) {
            ByteBuffer bb = ByteBuffer.allocate(2);
            bb.putShort(0, value);
            return bb;
        }

        @Override
        public short deserializeNoBoxing(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return 0;
            }
            if (bytes.remaining() != 2) {
                throw new InvalidTypeException("Invalid 16-bits integer value, expecting 2 bytes but got " + bytes.remaining());
            }
            return bytes.getShort(bytes.position());
        }
    }

    private static class TinyIntCodec
    extends PrimitiveByteCodec {
        private static final TinyIntCodec instance = new TinyIntCodec();

        private TinyIntCodec() {
            super(DataType.tinyint());
        }

        @Override
        public Byte parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : Byte.valueOf(Byte.parseByte(value));
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse 8-bits int value from \"%s\"", value));
            }
        }

        @Override
        public String format(Byte value) {
            if (value == null) {
                return "NULL";
            }
            return Byte.toString(value);
        }

        @Override
        public ByteBuffer serializeNoBoxing(byte value, ProtocolVersion protocolVersion) {
            ByteBuffer bb = ByteBuffer.allocate(1);
            bb.put(0, value);
            return bb;
        }

        @Override
        public byte deserializeNoBoxing(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return 0;
            }
            if (bytes.remaining() != 1) {
                throw new InvalidTypeException("Invalid 8-bits integer value, expecting 1 byte but got " + bytes.remaining());
            }
            return bytes.get(bytes.position());
        }
    }

    private static class InetCodec
    extends TypeCodec<InetAddress> {
        private static final InetCodec instance = new InetCodec();

        private InetCodec() {
            super(DataType.inet(), InetAddress.class);
        }

        @Override
        public InetAddress parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            if ((value = value.trim()).charAt(0) != '\'' || value.charAt(value.length() - 1) != '\'') {
                throw new InvalidTypeException(String.format("inet values must be enclosed in single quotes (\"%s\")", value));
            }
            try {
                return InetAddress.getByName(value.substring(1, value.length() - 1));
            }
            catch (Exception e) {
                throw new InvalidTypeException(String.format("Cannot parse inet value from \"%s\"", value));
            }
        }

        @Override
        public String format(InetAddress value) {
            if (value == null) {
                return "NULL";
            }
            return "'" + value.getHostAddress() + "'";
        }

        @Override
        public ByteBuffer serialize(InetAddress value, ProtocolVersion protocolVersion) {
            return value == null ? null : ByteBuffer.wrap(value.getAddress());
        }

        @Override
        public InetAddress deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return null;
            }
            try {
                return InetAddress.getByAddress(Bytes.getArray(bytes));
            }
            catch (UnknownHostException e) {
                throw new InvalidTypeException("Invalid bytes for inet value, got " + bytes.remaining() + " bytes");
            }
        }
    }

    private static class FloatCodec
    extends PrimitiveFloatCodec {
        private static final FloatCodec instance = new FloatCodec();

        private FloatCodec() {
            super(DataType.cfloat());
        }

        @Override
        public Float parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : Float.valueOf(Float.parseFloat(value));
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse 32-bits float value from \"%s\"", value));
            }
        }

        @Override
        public String format(Float value) {
            if (value == null) {
                return "NULL";
            }
            return Float.toString(value.floatValue());
        }

        @Override
        public ByteBuffer serializeNoBoxing(float value, ProtocolVersion protocolVersion) {
            ByteBuffer bb = ByteBuffer.allocate(4);
            bb.putFloat(0, value);
            return bb;
        }

        @Override
        public float deserializeNoBoxing(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return 0.0f;
            }
            if (bytes.remaining() != 4) {
                throw new InvalidTypeException("Invalid 32-bits float value, expecting 4 bytes but got " + bytes.remaining());
            }
            return bytes.getFloat(bytes.position());
        }
    }

    private static class DoubleCodec
    extends PrimitiveDoubleCodec {
        private static final DoubleCodec instance = new DoubleCodec();

        private DoubleCodec() {
            super(DataType.cdouble());
        }

        @Override
        public Double parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : Double.valueOf(Double.parseDouble(value));
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse 64-bits double value from \"%s\"", value));
            }
        }

        @Override
        public String format(Double value) {
            if (value == null) {
                return "NULL";
            }
            return Double.toString(value);
        }

        @Override
        public ByteBuffer serializeNoBoxing(double value, ProtocolVersion protocolVersion) {
            ByteBuffer bb = ByteBuffer.allocate(8);
            bb.putDouble(0, value);
            return bb;
        }

        @Override
        public double deserializeNoBoxing(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return 0.0;
            }
            if (bytes.remaining() != 8) {
                throw new InvalidTypeException("Invalid 64-bits double value, expecting 8 bytes but got " + bytes.remaining());
            }
            return bytes.getDouble(bytes.position());
        }
    }

    private static class DecimalCodec
    extends TypeCodec<BigDecimal> {
        private static final DecimalCodec instance = new DecimalCodec();

        private DecimalCodec() {
            super(DataType.decimal(), BigDecimal.class);
        }

        @Override
        public BigDecimal parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : new BigDecimal(value);
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse decimal value from \"%s\"", value));
            }
        }

        @Override
        public String format(BigDecimal value) {
            if (value == null) {
                return "NULL";
            }
            return value.toString();
        }

        @Override
        public ByteBuffer serialize(BigDecimal value, ProtocolVersion protocolVersion) {
            if (value == null) {
                return null;
            }
            BigInteger bi = value.unscaledValue();
            int scale = value.scale();
            byte[] bibytes = bi.toByteArray();
            ByteBuffer bytes = ByteBuffer.allocate(4 + bibytes.length);
            bytes.putInt(scale);
            bytes.put(bibytes);
            bytes.rewind();
            return bytes;
        }

        @Override
        public BigDecimal deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return null;
            }
            if (bytes.remaining() < 4) {
                throw new InvalidTypeException("Invalid decimal value, expecting at least 4 bytes but got " + bytes.remaining());
            }
            bytes = bytes.duplicate();
            int scale = bytes.getInt();
            byte[] bibytes = new byte[bytes.remaining()];
            bytes.get(bibytes);
            BigInteger bi = new BigInteger(bibytes);
            return new BigDecimal(bi, scale);
        }
    }

    private static class BooleanCodec
    extends PrimitiveBooleanCodec {
        private static final ByteBuffer TRUE = ByteBuffer.wrap(new byte[]{1});
        private static final ByteBuffer FALSE = ByteBuffer.wrap(new byte[]{0});
        private static final BooleanCodec instance = new BooleanCodec();

        private BooleanCodec() {
            super(DataType.cboolean());
        }

        @Override
        public Boolean parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            if (value.equalsIgnoreCase(Boolean.FALSE.toString())) {
                return false;
            }
            if (value.equalsIgnoreCase(Boolean.TRUE.toString())) {
                return true;
            }
            throw new InvalidTypeException(String.format("Cannot parse boolean value from \"%s\"", value));
        }

        @Override
        public String format(Boolean value) {
            if (value == null) {
                return "NULL";
            }
            return value != false ? "true" : "false";
        }

        @Override
        public ByteBuffer serializeNoBoxing(boolean value, ProtocolVersion protocolVersion) {
            return value ? TRUE.duplicate() : FALSE.duplicate();
        }

        @Override
        public boolean deserializeNoBoxing(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return false;
            }
            if (bytes.remaining() != 1) {
                throw new InvalidTypeException("Invalid boolean value, expecting 1 byte but got " + bytes.remaining());
            }
            return bytes.get(bytes.position()) != 0;
        }
    }

    private static class CustomCodec
    extends TypeCodec<ByteBuffer> {
        private CustomCodec(DataType custom) {
            super(custom, ByteBuffer.class);
            assert (custom.getName() == DataType.Name.CUSTOM);
        }

        @Override
        public ByteBuffer parse(String value) {
            return value == null || value.isEmpty() || value.equals("NULL") ? null : Bytes.fromHexString(value);
        }

        @Override
        public String format(ByteBuffer value) {
            if (value == null) {
                return "NULL";
            }
            return Bytes.toHexString(value);
        }

        @Override
        public ByteBuffer serialize(ByteBuffer value, ProtocolVersion protocolVersion) {
            return value == null ? null : value.duplicate();
        }

        @Override
        public ByteBuffer deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null ? null : bytes.duplicate();
        }
    }

    private static class BlobCodec
    extends TypeCodec<ByteBuffer> {
        private static final BlobCodec instance = new BlobCodec();

        private BlobCodec() {
            super(DataType.blob(), ByteBuffer.class);
        }

        @Override
        public ByteBuffer parse(String value) {
            return value == null || value.isEmpty() || value.equals("NULL") ? null : Bytes.fromHexString(value);
        }

        @Override
        public String format(ByteBuffer value) {
            if (value == null) {
                return "NULL";
            }
            return Bytes.toHexString(value);
        }

        @Override
        public ByteBuffer serialize(ByteBuffer value, ProtocolVersion protocolVersion) {
            return value == null ? null : value.duplicate();
        }

        @Override
        public ByteBuffer deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null ? null : bytes.duplicate();
        }
    }

    private static class CounterCodec
    extends LongCodec {
        private static final CounterCodec instance = new CounterCodec();

        private CounterCodec() {
            super(DataType.counter());
        }
    }

    private static class BigintCodec
    extends LongCodec {
        private static final BigintCodec instance = new BigintCodec();

        private BigintCodec() {
            super(DataType.bigint());
        }
    }

    private static abstract class LongCodec
    extends PrimitiveLongCodec {
        private LongCodec(DataType cqlType) {
            super(cqlType);
        }

        @Override
        public Long parse(String value) {
            try {
                return value == null || value.isEmpty() || value.equals("NULL") ? null : Long.valueOf(Long.parseLong(value));
            }
            catch (NumberFormatException e) {
                throw new InvalidTypeException(String.format("Cannot parse 64-bits long value from \"%s\"", value));
            }
        }

        @Override
        public String format(Long value) {
            if (value == null) {
                return "NULL";
            }
            return Long.toString(value);
        }

        @Override
        public ByteBuffer serializeNoBoxing(long value, ProtocolVersion protocolVersion) {
            ByteBuffer bb = ByteBuffer.allocate(8);
            bb.putLong(0, value);
            return bb;
        }

        @Override
        public long deserializeNoBoxing(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null || bytes.remaining() == 0) {
                return 0L;
            }
            if (bytes.remaining() != 8) {
                throw new InvalidTypeException("Invalid 64-bits long value, expecting 8 bytes but got " + bytes.remaining());
            }
            return bytes.getLong(bytes.position());
        }
    }

    private static class AsciiCodec
    extends StringCodec {
        private static final AsciiCodec instance = new AsciiCodec();
        private static final Pattern ASCII_PATTERN = Pattern.compile("^\\p{ASCII}*$");

        private AsciiCodec() {
            super(DataType.ascii(), Charset.forName("US-ASCII"));
        }

        @Override
        public ByteBuffer serialize(String value, ProtocolVersion protocolVersion) {
            if (value != null && !ASCII_PATTERN.matcher(value).matches()) {
                throw new InvalidTypeException(String.format("%s is not a valid ASCII String", value));
            }
            return super.serialize(value, protocolVersion);
        }

        @Override
        public String format(String value) {
            if (value != null && !ASCII_PATTERN.matcher(value).matches()) {
                throw new InvalidTypeException(String.format("%s is not a valid ASCII String", value));
            }
            return super.format(value);
        }
    }

    private static class VarcharCodec
    extends StringCodec {
        private static final VarcharCodec instance = new VarcharCodec();

        private VarcharCodec() {
            super(DataType.varchar(), Charset.forName("UTF-8"));
        }
    }

    private static abstract class StringCodec
    extends TypeCodec<String> {
        private final Charset charset;

        private StringCodec(DataType cqlType, Charset charset) {
            super(cqlType, String.class);
            this.charset = charset;
        }

        @Override
        public String parse(String value) {
            if (value == null || value.isEmpty() || value.equals("NULL")) {
                return null;
            }
            if (value.charAt(0) != '\'' || value.charAt(value.length() - 1) != '\'') {
                throw new InvalidTypeException("text or varchar values must be enclosed by single quotes");
            }
            return value.substring(1, value.length() - 1).replace("''", "'");
        }

        @Override
        public String format(String value) {
            if (value == null) {
                return "NULL";
            }
            return '\'' + StringCodec.replace(value, '\'', "''") + '\'';
        }

        private static String replace(String text, char search, String replacement) {
            if (text == null || text.isEmpty()) {
                return text;
            }
            int nbMatch = 0;
            int start = -1;
            do {
                if ((start = text.indexOf(search, start + 1)) == -1) continue;
                ++nbMatch;
            } while (start != -1);
            if (nbMatch == 0) {
                return text;
            }
            int newLength = text.length() + nbMatch * (replacement.length() - 1);
            char[] result = new char[newLength];
            int newIdx = 0;
            for (int i = 0; i < text.length(); ++i) {
                char c = text.charAt(i);
                if (c == search) {
                    for (int r = 0; r < replacement.length(); ++r) {
                        result[newIdx++] = replacement.charAt(r);
                    }
                    continue;
                }
                result[newIdx++] = c;
            }
            return new String(result);
        }

        @Override
        public ByteBuffer serialize(String value, ProtocolVersion protocolVersion) {
            return value == null ? null : ByteBuffer.wrap(value.getBytes(this.charset));
        }

        @Override
        public String deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            if (bytes == null) {
                return null;
            }
            if (bytes.remaining() == 0) {
                return "";
            }
            return new String(Bytes.getArray(bytes), this.charset);
        }
    }

    public static abstract class PrimitiveDoubleCodec
    extends TypeCodec<Double> {
        protected PrimitiveDoubleCodec(DataType cqlType) {
            super(cqlType, Double.class);
        }

        public abstract ByteBuffer serializeNoBoxing(double var1, ProtocolVersion var3);

        public abstract double deserializeNoBoxing(ByteBuffer var1, ProtocolVersion var2);

        @Override
        public ByteBuffer serialize(Double value, ProtocolVersion protocolVersion) {
            return value == null ? null : this.serializeNoBoxing(value, protocolVersion);
        }

        @Override
        public Double deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : Double.valueOf(this.deserializeNoBoxing(bytes, protocolVersion));
        }
    }

    public static abstract class PrimitiveFloatCodec
    extends TypeCodec<Float> {
        protected PrimitiveFloatCodec(DataType cqlType) {
            super(cqlType, Float.class);
        }

        public abstract ByteBuffer serializeNoBoxing(float var1, ProtocolVersion var2);

        public abstract float deserializeNoBoxing(ByteBuffer var1, ProtocolVersion var2);

        @Override
        public ByteBuffer serialize(Float value, ProtocolVersion protocolVersion) {
            return value == null ? null : this.serializeNoBoxing(value.floatValue(), protocolVersion);
        }

        @Override
        public Float deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : Float.valueOf(this.deserializeNoBoxing(bytes, protocolVersion));
        }
    }

    public static abstract class PrimitiveLongCodec
    extends TypeCodec<Long> {
        protected PrimitiveLongCodec(DataType cqlType) {
            super(cqlType, Long.class);
        }

        public abstract ByteBuffer serializeNoBoxing(long var1, ProtocolVersion var3);

        public abstract long deserializeNoBoxing(ByteBuffer var1, ProtocolVersion var2);

        @Override
        public ByteBuffer serialize(Long value, ProtocolVersion protocolVersion) {
            return value == null ? null : this.serializeNoBoxing(value, protocolVersion);
        }

        @Override
        public Long deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : Long.valueOf(this.deserializeNoBoxing(bytes, protocolVersion));
        }
    }

    public static abstract class PrimitiveIntCodec
    extends TypeCodec<Integer> {
        protected PrimitiveIntCodec(DataType cqlType) {
            super(cqlType, Integer.class);
        }

        public abstract ByteBuffer serializeNoBoxing(int var1, ProtocolVersion var2);

        public abstract int deserializeNoBoxing(ByteBuffer var1, ProtocolVersion var2);

        @Override
        public ByteBuffer serialize(Integer value, ProtocolVersion protocolVersion) {
            return value == null ? null : this.serializeNoBoxing(value, protocolVersion);
        }

        @Override
        public Integer deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : Integer.valueOf(this.deserializeNoBoxing(bytes, protocolVersion));
        }
    }

    public static abstract class PrimitiveShortCodec
    extends TypeCodec<Short> {
        protected PrimitiveShortCodec(DataType cqlType) {
            super(cqlType, Short.class);
        }

        public abstract ByteBuffer serializeNoBoxing(short var1, ProtocolVersion var2);

        public abstract short deserializeNoBoxing(ByteBuffer var1, ProtocolVersion var2);

        @Override
        public ByteBuffer serialize(Short value, ProtocolVersion protocolVersion) {
            return value == null ? null : this.serializeNoBoxing(value, protocolVersion);
        }

        @Override
        public Short deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : Short.valueOf(this.deserializeNoBoxing(bytes, protocolVersion));
        }
    }

    public static abstract class PrimitiveByteCodec
    extends TypeCodec<Byte> {
        protected PrimitiveByteCodec(DataType cqlType) {
            super(cqlType, Byte.class);
        }

        public abstract ByteBuffer serializeNoBoxing(byte var1, ProtocolVersion var2);

        public abstract byte deserializeNoBoxing(ByteBuffer var1, ProtocolVersion var2);

        @Override
        public ByteBuffer serialize(Byte value, ProtocolVersion protocolVersion) {
            return value == null ? null : this.serializeNoBoxing(value, protocolVersion);
        }

        @Override
        public Byte deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : Byte.valueOf(this.deserializeNoBoxing(bytes, protocolVersion));
        }
    }

    public static abstract class PrimitiveBooleanCodec
    extends TypeCodec<Boolean> {
        protected PrimitiveBooleanCodec(DataType cqlType) {
            super(cqlType, Boolean.class);
        }

        public abstract ByteBuffer serializeNoBoxing(boolean var1, ProtocolVersion var2);

        public abstract boolean deserializeNoBoxing(ByteBuffer var1, ProtocolVersion var2);

        @Override
        public ByteBuffer serialize(Boolean value, ProtocolVersion protocolVersion) {
            return value == null ? null : this.serializeNoBoxing(value, protocolVersion);
        }

        @Override
        public Boolean deserialize(ByteBuffer bytes, ProtocolVersion protocolVersion) {
            return bytes == null || bytes.remaining() == 0 ? null : Boolean.valueOf(this.deserializeNoBoxing(bytes, protocolVersion));
        }
    }
}

