# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

import construct
from collections import namedtuple


class RequestID:
    Handshake          = 0x848129F9E6D41225
    AppendFile         = 0xB39BE649352837A0
    ReplaceFile        = 0x3F77F91A127042A4
    ReadFile           = 0xCD4348AA14506957
    SparseFile         = 0x9829A4998973610A
    RenameFile         = 0x06E3BC0C13FB4D1A
    SwapFiles          = 0xA6332FF2AC61EFD6
    DeleteFile         = 0xA164C38B156B6F33
    GetFileSize        = 0x4B77D3A9F86546F7
    GetFileInfo        = 0x03C0619BE3F3C540
    GetFileInfoList    = 0x47FA157DF7D1CCB7
    LockFile           = 0x75197F49275C44B9
    UnlockFile         = 0xAE9DD83CB44E4ACA
    GetQuota           = 0xF498EAF781FDE6EC
    GetAllocationInfo  = 0xCB18EF28539E65B6
    ExtendedOpenCreate = 0x56525704DD904E10


class ServerErrorCodes:
    UnknownRequest    = 0x0004
    FileNotFound      = 0x0028
    QuotaExceeded     = 0x005d
    FileNotLocked     = 0x0068
    ReadOnlyAccess    = 0x006a
    NotSupported      = 0x0072
    FileLocked        = 0x0410
    InvalidLock       = 0x0411
    FileAlreadyExists = 0x041a


class RequestType:
    Short  = 0
    Input  = 1
    Output = 2


class LockType:
    Shared    = 0
    Normal    = 1
    Exclusive = 2
    NoLock    = 3


class RedirectFormat:
    NoRedirect = 0
    DnsName    = 1


class QuotaLimits:
    Unknown  = 0xffffffffffffffff
    Infinity = 0xffffffffffffffff - 1


class ExtendedOpenCreateFlags:
    Create   = 1
    Truncate = 2
    NewFile  = 4


def String(name):
    return construct.PascalString(name, length_field=construct.ULInt32('length'), encoding='utf-8')


def FileTime(name):
    return construct.ExprAdapter(  # Converts integer microseconds <=> float seconds
        construct.ULInt64(name),
        encoder=lambda obj, ctx: int(obj * 1000000),
        decoder=lambda obj, ctx: obj / 1000000.
    )


RequestHeader = construct.Struct(
    'request_header',
    construct.ULInt64('id'),
    construct.ULInt64('serial')
)

StreamingRequestHeader = construct.Struct(
    'streaming_request_header',
    construct.ULInt64('id')
)

ResponseHeader = construct.Struct(
    'response_header',
    construct.ULInt32('result'),
    construct.ULInt64('serial')
)

StreamingResponseHeader = construct.Struct(
    'streaming_response_header',
    construct.ULInt32('result')
)

VersionInfo = construct.Struct(
    'version_info',
    String('app_name'),
    construct.ULInt32('major_version'),
    construct.ULInt32('minor_version'),
    construct.ULInt32('build_number')
)

CapabilitiesInfo = construct.Struct(
    'capabilities_info',
    construct.ULInt32('bytes_count'),
    construct.Array(lambda ctx: ctx.bytes_count, construct.ULInt8('value'))
)

FileInfo = construct.Struct(
    'file_info',
    FileTime('last_change_time'),
    construct.ULInt64('logical_size'),
    construct.ULInt64('physical_size')
)

FileInfoListHeader = construct.Struct(
    'file_info_list_header',
    construct.ULInt32('items_count')
)

FileInfoListItem = construct.Struct(
    'file_info_list_item',
    String('name'),
    construct.Embedded(FileInfo)
)

InputChunk = construct.Struct(
    'input_chunk',
    construct.ULInt32('_attr'),
    construct.Value('size', lambda ctx: ctx._attr & 0x7fffffff),
    construct.Value('is_last', lambda ctx: bool(ctx._attr & 0x80000000)),
    construct.Bytes('data', lambda ctx: ctx.size)
)


def RequestHeaderFactory(req_id):
    def encoder(obj, ctx):
        obj.id = req_id
        return obj

    def decoder(obj, ctx):
        return obj

    return construct.Embedded(construct.ExprAdapter(RequestHeader, encoder=encoder, decoder=decoder))


def StreamingRequestHeaderFactory(req_id):
    def encoder(obj, ctx):
        obj.id = req_id
        return obj

    def decoder(obj, ctx):
        return obj

    return construct.Embedded(construct.ExprAdapter(StreamingRequestHeader, encoder=encoder, decoder=decoder))


class BaseBuilder:
    def __init__(self, builder):
        self._builder = builder

    def __call__(self):
        return self

    def build(self, data):
        return self._builder.build(data)


def safe_parse_stream(parser, stream):
    save_stream_pos = stream.tell()
    try:
        return parser.parse_stream(stream)
    except construct.ConstructError:
        stream.seek(save_stream_pos)
        raise


class BaseParser:
    def __init__(self, parser):
        self._parser = parser

    def __call__(self):
        return self

    def parse_stream(self, stream):
        return safe_parse_stream(self._parser, stream)


class FileInfoListParser:
    def __init__(self):
        self._ctx = None
        self._buffer_pos = None

    def parse_stream(self, stream):
        if self._ctx is None:
            self._ctx = FileInfoListHeader.parse_stream(stream)
            self._buffer_pos = stream.tell()
        stream.seek(self._buffer_pos)

        if self._ctx.get('items') is None:
            self._ctx['items'] = construct.ListContainer()

        items_count = self._ctx.items_count
        parsed_items_count = len(self._ctx['items'])
        for i in range(0, items_count - parsed_items_count):
            self._ctx['items'].append(FileInfoListItem.parse_stream(stream))
            self._buffer_pos = stream.tell()

        return self._ctx


RequestInfo = namedtuple('RequestInfo', ('builder_factory', 'parser_factory', 'type'))

Requests = dict()

Requests[RequestID.Handshake] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'handshake_request',
            RequestHeaderFactory(RequestID.Handshake),
            String('security_token'),
            VersionInfo,
            CapabilitiesInfo
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'handshake_response',
            VersionInfo,
            CapabilitiesInfo,
            construct.ULInt16('redirect_type'),
            construct.ULInt32('redirect_size'),
            construct.Bytes('redirect_data', lambda ctx: ctx.redirect_size)
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.AppendFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'append_file_request',
            StreamingRequestHeaderFactory(RequestID.AppendFile),
            String('name'),
            String('prefix'),
            construct.ULInt64('lock_id')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'append_file_response',
            construct.ULInt32('sample_size'),
            construct.ULInt32('first_sample_size'),
            construct.ULInt64('start_offset'),
            construct.ULInt64('quota_left')
        )
    ),
    type=RequestType.Output
)

Requests[RequestID.ReplaceFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'replace_file_request',
            StreamingRequestHeaderFactory(RequestID.ReplaceFile),
            String('name'),
            String('prefix'),
            construct.ULInt64('lock_id')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'replace_file_response',
            construct.ULInt32('sample_size'),
            construct.ULInt64('quota_left')
        )
    ),
    type=RequestType.Output
)

Requests[RequestID.ReadFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'read_file_request',
            StreamingRequestHeaderFactory(RequestID.ReadFile),
            String('name'),
            String('prefix'),
            construct.ULInt64('offset'),
            construct.ULInt64('size')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'read_file_response',
            construct.ULInt32('sample_size')
        )
    ),
    type=RequestType.Input
)

Requests[RequestID.SparseFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'sparse_file_request',
            RequestHeaderFactory(RequestID.SparseFile),
            String('name'),
            String('prefix'),
            construct.ULInt64('lock_id'),
            construct.ULInt32('offsets_count'),
            construct.Array(lambda ctx: ctx.offsets_count, construct.ULInt64('offsets'))
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'sparse_file_response',
            # No data in response
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.RenameFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'rename_file_request',
            RequestHeaderFactory(RequestID.RenameFile),
            String('old_name'),
            String('new_name'),
            String('old_prefix'),
            String('new_prefix'),
            construct.ULInt64('lock_id')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'rename_file_response'
            # No data in response
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.SwapFiles] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'swap_file_request',
            RequestHeaderFactory(RequestID.SwapFiles),
            String('one_name'),
            String('two_name'),
            String('one_prefix'),
            String('two_prefix'),
            construct.ULInt64('one_lock_id'),
            construct.ULInt64('two_lock_id')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'swap_file_response'
            # No data in response
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.DeleteFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'delete_file_request',
            RequestHeaderFactory(RequestID.DeleteFile),
            String('name'),
            String('prefix'),
            construct.ULInt64('lock_id')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'delete_file_response'
            # No data in response
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.GetFileSize] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'get_file_size_request',
            RequestHeaderFactory(RequestID.GetFileSize),
            String('name'),
            String('prefix')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'get_file_size_response',
            construct.ULInt64('size')
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.GetFileInfo] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'get_file_info_request',
            RequestHeaderFactory(RequestID.GetFileInfo),
            String('name'),
            String('prefix')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'get_file_info_response',
            construct.Embedded(FileInfo)
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.GetFileInfoList] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'get_file_info_list_request',
            RequestHeaderFactory(RequestID.GetFileInfoList),
            String('prefix')
        )
    ),
    parser_factory=lambda: FileInfoListParser(),
    type=RequestType.Short
)

Requests[RequestID.LockFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'lock_file_request',
            RequestHeaderFactory(RequestID.LockFile),
            String('name'),
            String('prefix'),
            construct.ULInt64('lock_id'),
            construct.ULInt8('lock_type')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'lock_file_response',
            construct.ULInt64('lock_id')
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.UnlockFile] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'unlock_file_request',
            RequestHeaderFactory(RequestID.UnlockFile),
            String('name'),
            String('prefix'),
            construct.ULInt64('lock_id')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'unlock_file_response',
            # No data in response
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.GetQuota] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'get_quota_request',
            RequestHeaderFactory(RequestID.GetQuota),
            String('prefix')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'get_quota_response',
            construct.ULInt64('data_hard_limit'),
            construct.ULInt64('data_soft_limit'),
            construct.ULInt64('data_current'),
            construct.ULInt64('inode_hard_limit'),
            construct.ULInt64('inode_soft_limit'),
            construct.ULInt64('inode_current')
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.GetAllocationInfo] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'get_allocation_info_request',
            RequestHeaderFactory(RequestID.GetAllocationInfo),
            String('name'),
            String('prefix'),
            construct.ULInt64('offset')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'get_allocation_info_response',
            construct.ULInt32('fragment_size'),
            construct.ULInt32('block_size'),
            construct.ULInt32('allocations_count'),
            construct.Array(
                lambda ctx: ctx.allocations_count,
                construct.ULInt32('fragment_allocations')
            )
        )
    ),
    type=RequestType.Short
)

Requests[RequestID.ExtendedOpenCreate] = RequestInfo(
    builder_factory=BaseBuilder(
        construct.Struct(
            'extended_open_create_request',
            RequestHeaderFactory(RequestID.ExtendedOpenCreate),
            String('name'),
            String('prefix'),
            construct.ULInt8('flags'),
            construct.ULInt64('lock_id'),
            construct.ULInt8('lock_type')
        )
    ),
    parser_factory=BaseParser(
        construct.Struct(
            'extended_open_create_response',
            construct.ULInt64('lock_id'),
            construct.Embedded(FileInfo)
        )
    ),
    type=RequestType.Short
)

__all__ = (
    'RequestID',
    'ServerErrorCodes',
    'RequestType',
    'LockType',
    'RedirectFormat',
    'QuotaLimits',
    'ExtendedOpenCreateFlags',
    'RequestHeader',
    'StreamingRequestHeader',
    'ResponseHeader',
    'StreamingResponseHeader',
    'InputChunk',
    'Requests'
)
