# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

from .requests import ServerErrorCodes as errors


class BaseError(Exception):
    def __init__(self, host, port, message=None):
        self.host = host
        self.port = port
        self.message = message


class ConnectError(BaseError):
    def __init__(self, host, port, connect_exc=None):
        super().__init__(host, port, 'Cannot connect to the server')
        self.connect_exc = connect_exc


class ServerError(BaseError):
    MESSAGE = 'Server returned error on request'
    def __init__(self, host, port, request, error_code, message=None):
        super().__init__(host, port, message or self.MESSAGE)
        self.request = request
        self.error_code = error_code


class UnsupportedRequestError(ServerError):
    MESSAGE = 'Unknown or unsupported request for server'


class FileNotFoundError(ServerError):
    MESSAGE = 'Requested file not found on server'


class FileAlreadyExistsError(ServerError):
    MESSAGE = 'Requested file already existed on server'


class FileLockedError(ServerError):
    MESSAGE = 'Requested file already locked'


class FileNotLockedError(ServerError):
    MESSAGE = 'Request cannot be processed without file lock'


class InvalidLockError(ServerError):
    MESSAGE = 'Invalid file lock'


class QuotaExceededError(ServerError):
    MESSAGE = 'Storage quota exceeded'


class ReadOnlyAccessError(ServerError):
    MESSAGE = 'Cannot perform request due to read only access to server'


errors_map = {
    errors.UnknownRequest: UnsupportedRequestError,
    errors.FileNotFound: FileNotFoundError,
    errors.QuotaExceeded: QuotaExceededError,
    errors.FileNotLocked: FileNotLockedError,
    errors.ReadOnlyAccess: ReadOnlyAccessError,
    errors.NotSupported: UnsupportedRequestError,
    errors.FileLocked: FileLockedError,
    errors.InvalidLock: InvalidLockError,
    errors.FileAlreadyExists: FileAlreadyExistsError
}


def make_server_error(host, port, request, error_code):
    pure_error_code = error_code & 0x7fff
    error_class = errors_map.get(pure_error_code, ServerError)
    return error_class(host, port, request, error_code)


__all__ = (
    'BaseError',
    'ConnectError',
    'ServerError',
    'UnsupportedRequestError',
    'FileNotFoundError',
    'FileAlreadyExistsError',
    'FileLockedError',
    'FileNotLockedError',
    'InvalidLockError',
    'QuotaExceededError',
    'ReadOnlyAccessError',
    'make_server_error'
)
