# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

from __future__ import print_function
from acronis.lib import axmlrpc
from collections import OrderedDict
import hashlib
import logging


class RPCrequest(object):
    def __init__(self, **kwargs):
        self.args = kwargs

    def create(self):
        request = OrderedDict({'Action': self.request_name()})
        requestPayload = self.payload()
        logging.debug('RPC request parameters: %s', requestPayload)
        request['Sign'] = self.create_sign(requestPayload)
        request['Data'] = requestPayload
        return axmlrpc.dumps(request).encode('utf-8')

    def request_name(self):
        return self.__class__.__name__.replace('Request', '')

    def create_sign(self, payload):
        source = '#'.join(['fTY4,Q118', self.request_name(), '#'.join(payload[key] for key in sorted(payload.keys()))])
        return hashlib.md5(source.encode('utf-8')).hexdigest()

    def payload(self):
        result = {'BrandId': '1', 'Client': 'RPC command line', 'EmailAddress': self.args['email']}
        if self.args.get('account'):
            result.update({'AccountId': str(self.args['account'])})
        if self.args.get('service'):
            result.update({'Service': str(self.args['service'])})
        return result


class PasswordProtectedRequest(RPCrequest):
    def payload(self):
        if self.args.get('token'):
            password = 'token::' + self.args['token']
        elif self.args.get('password_hash'):
            password = self.args['password_hash']
        else:
            password = hashlib.md5(self.args['password'].encode('ascii')).hexdigest()
        result = super(PasswordProtectedRequest, self).payload()
        result.update({'Password': password})
        return result


class GetSubaccountsListRequest(PasswordProtectedRequest):
    def payload(self):
        result = super(GetSubaccountsListRequest, self).payload()
        if self.args.get('machine'):
            result.update({'MachineId': str(self.args['machine'])})
        return result


class CheckAccountRequest(PasswordProtectedRequest): pass
class GetRegistrationsRequest(PasswordProtectedRequest): pass


class GetRedirectRequest(PasswordProtectedRequest):
    def payload(self):
        result = super(GetRedirectRequest, self).payload()
        result.update({'Subaccount': str(self.args.get('subaccountId') or 0)})
        return result


class GetMachineTokenRequest(GetRedirectRequest): pass


class UpdateSubaccountsRequest(PasswordProtectedRequest):
    def payload(self):
        result = super(UpdateSubaccountsRequest, self).payload()
        result.update({'Subaccounts': {'Subaccount': {
            'id': str(self.args['id']),
            'Name': str(self.args['name']),
            'Description': str(self.args['description']),
            'AutoNextPP': str(self.args['autoNextPP']),
        }}})
        return result


class UpdateStorageSizeRequest(PasswordProtectedRequest):
    def payload(self):
        result = super(UpdateStorageSizeRequest, self).payload()
        result.update({'Subaccount': str(self.args['subaccountId'])})
        return result


class GetUsedSpaceRequest(UpdateStorageSizeRequest): pass


class UpdateSubaccountRegistrationStatusRequest(PasswordProtectedRequest):
    def payload(self):
        result = super(UpdateSubaccountRegistrationStatusRequest, self).payload()
        result.update({'RegistrationId': str(self.args['orderId'])})
        result.update({'StatusId': str(self.args['status'])})
        result.update({'Subaccount': str(self.args['subaccountId'])})
        return result
