#!/usr/bin/env python
# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

from __future__ import print_function
from argparse import ArgumentParser
from json import dumps as pretty_print
from six.moves.urllib.error import URLError
from six.moves.urllib.request import HTTPSHandler, build_opener
from ssl import create_default_context, CERT_NONE
from time import sleep
import logging
import rpc_request
import rpc_response
import sys


def retry_if(count, delay, error_class):
    def decorator(func):
        def wrapper(*args, **kwargs):
            err = None
            for i in range(count):
                try:
                    return func(*args, **kwargs)
                except error_class as e:
                    if getattr(e, 'reason', None) and getattr(e.reason, 'reason', '') == 'CERTIFICATE_VERIFY_FAILED':
                        raise e
                    logging.info('%s, retry %s/%s', e, i, count)
                    sleep(delay)
                    err = e
            raise err
        return wrapper
    return decorator

@retry_if(count=5, delay=20, error_class=URLError)
#  Basically it is the same as urllib.request.urlopen but with a custom User-agent.
#  Plus the built-in urlopen supports context parameter since python 3.4.3 only.
def rpc_opener(url, request, context=None):
    https_handler = HTTPSHandler(context=context)
    opener = build_opener(https_handler)
    opener.addheaders = [('User-agent', 'RPC command line/2.0')]
    return opener.open(url, request)

def execute_rpc_request(url, request_name, response_decoder=rpc_response.decode_plain_response, opener=rpc_opener, insecure=False, **kwargs):
    try:
        ClassType = getattr(rpc_request, request_name + 'Request')
    except Exception as ex:
        raise RuntimeError('Invalid request: {0}'.format(request_name))
    request_instance = ClassType(**kwargs)
    request = request_instance.create()
    logging.debug('RPC %s request: %s', url, request)
    context = kwargs.get('context', create_default_context())
    if 'context' not in kwargs and insecure:
        context.check_hostname = False
        context.verify_mode = CERT_NONE
    response = opener(url, request, context=context)
    response = rpc_response.check_response_status(response)
    return response_decoder(response)

def _execute_rpc_request(request_name, args, response_decoder=rpc_response.decode_plain_response):
    url = args.__dict__.pop('url')
    return execute_rpc_request(url, request_name, response_decoder=response_decoder, **args.__dict__)

def create_arguments_parser():
    TestServerURL = 'http://rpc.webtest-2.acronis.ru:80/online-backup/2.0/'
    parser = ArgumentParser(description='execute requests to rpc')
    parser.add_argument('--version', action='version', version='1.1')
    parser.add_argument('-s', '--server', dest='url', default=TestServerURL, help='RPC server URL (default: webtest)')
    parser.add_argument('-e', '--email', dest='email', help='Account e-mail', required=True)
    parser.add_argument('-p', '--pass', dest='password', help='Password', metavar='PASS')
    parser.add_argument('-a', '--passhash', dest='password_hash', help='Password hash')
    parser.add_argument('-t', '--token', dest='token', help='Access token')
    parser.add_argument('--secure', dest='insecure', action='store_false', help='Forces SSL certificate validation')
    parser.add_argument('--insecure', dest='insecure', action='store_true', help='Allows connection to a server with not trusted SSL certificate')
    parser.set_defaults(insecure=False)
    subparsers = parser.add_subparsers()

    parser_command1  = subparsers.add_parser('redirect', help='request main server address')
    parser_command1.add_argument('-b', '--box', dest='subaccountId', type=int, help='Box ID', metavar='ID', required=False)
    parser_command1.set_defaults(func=lambda args: _execute_rpc_request('GetRedirect', args, rpc_response.decode_redirect_response))

    parser_command2 = subparsers.add_parser('registrations', help='list all account registrations')
    parser_command2.set_defaults(func=lambda args: _execute_rpc_request('GetRegistrations', args,
                                                    lambda response: pretty_print(rpc_response.decode_registrations_response(response), indent=2)))

    parser_command3 = subparsers.add_parser('update', help='update subaccount registration status')
    parser_command3.add_argument('-b', '--box', dest='subaccountId', type=int, help='Box ID', metavar='ID', required=True)
    parser_command3.add_argument('-r', '--order', dest='orderId', type=int, help='Registration ID', required=True)
    parser_command3.add_argument('-t', '--status', dest='status', type=int, help='Registration status', required=True)
    parser_command3.set_defaults(func=lambda args: _execute_rpc_request('UpdateSubaccountRegistrationStatus', args, lambda _: 'OK'))

    parser_command4 = subparsers.add_parser('resize', help='update account size')
    parser_command4.add_argument('-b', '--box', dest='subaccountId', type=int, help='Box ID', metavar='ID', required=True)
    parser_command4.set_defaults(func=lambda args: _execute_rpc_request('UpdateStorageSize', args))

    parser_command5 = subparsers.add_parser('list', help='list subaccounts')
    parser_command5.set_defaults(func=lambda args: _execute_rpc_request('GetSubaccountsList', args, rpc_response.decode_getSubaccountsList_response))

    parser_command6 = subparsers.add_parser('check', help='check account')
    parser_command6.set_defaults(func=lambda args: _execute_rpc_request('CheckAccount', args, lambda _: 'OK'))

    parser_command7 = subparsers.add_parser('size', help='check account used size')
    parser_command7.add_argument('-b', '--box', dest='subaccountId', type=int, help='Box ID', metavar='ID', required=True)
    parser_command7.set_defaults(func=lambda args: _execute_rpc_request('GetUsedSpace', args, rpc_response.decode_getUsedSpace_response))

    parser_command8 = subparsers.add_parser('token', help='request access token')
    parser_command8.add_argument('-b', '--box', dest='subaccountId', type=int, help='Box ID', metavar='ID', required=False)
    parser_command8.set_defaults(func=lambda args: _execute_rpc_request('GetMachineToken', args, rpc_response.decode_token_response))
    return parser

def main():
    from acronis.lib import axmlrpc
    try:
        parser = create_arguments_parser()
        args = parser.parse_args()
        print(args.func(args))
    except IOError as ex:
        logging.debug(ex, exc_info=True)
        print(ex, file=sys.stderr)
        sys.exit(3)
    except axmlrpc.Error as ex:
        logging.debug(ex, exc_info=True)
        print('Error: {0}'.format(ex.description), file=sys.stderr)
        sys.exit(2)
    except Exception as ex:
        logging.debug(ex, exc_info=True)
        print(ex, file=sys.stderr)
        sys.exit(1)

if '__main__' == __name__:
  main()
