# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

import json
import logging
import logging.config
import os
from colorlog import ColoredFormatter
from colorlog.escape_codes import parse_colors


def paint_string(string, color):
    return parse_colors(color) + string + parse_colors('reset')


class ArgumentsPainter(ColoredFormatter):
    def __init__(self, *args, **kwargs):
        self.argument_colors = kwargs.pop('argument_colors', {'INFO': 'cyan'})
        super(ArgumentsPainter, self).__init__(*args, **kwargs)

    def format(self, record):
        color = self.argument_colors.get(record.levelname, None)
        if color is not None:
            record.args = tuple([paint_string(str(argument), color) for argument in record.args])
        return super(ArgumentsPainter, self).format(record)


def default_config_path():
    return os.path.join(os.path.dirname(__file__), 'logging.json')

def setup_logging(default_path=default_config_path(), default_level=logging.INFO, env_key='LOG_CFG'):
    path = os.getenv(env_key, None) or default_path
    if os.path.exists(path):
        with open(path, 'rt') as f:
            logging.config.dictConfig(json.load(f))
    else:
        logging.basicConfig(level=default_level)
        format = "%(log_color)s%(message)s"
        log_colors = {'ERROR': 'bold_red', 'WARNING': 'yellow'}
        formatter = ArgumentsPainter(format, log_colors=log_colors)
        logging.getLogger().handlers[0].setFormatter(formatter)
