#!/usr/bin/env python
# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

import is_lsr
import logging
import os
import subprocess
from baas_lsr_support import find_machine_backups, CommonBoxName
from colorama import init as colorama_init
from colorlog.escape_codes import parse_colors
from is_lsr import configuration
from xml.etree import ElementTree as ET


class NoOrderError(is_lsr.Error):
    def __init__(self, text):
        self.text = text

    def __str__(self):
        return 'No Large Scale Recovery orders are ready for processing: ' + self.text

    @staticmethod
    def create(registrations, box):
        order = [registration for registration in registrations if lsr_order(registration, box)]
        if order:
            return NoOrderError('the subaccount {0} order is in state {1}'.format(box, order[0]['StatusId']))
        else:
            return NoOrderError('no orders available for the subaccount ' + box)

def lsr_order(registration, box):
    return registration['ProductType'] == 'LSR' and registration['SubaccountId'] == box

def suitable_for_lsr(registration, box):
    return lsr_order(registration, box) and registration['StatusId'] in ['103', '106']

def mark_as_lsr(file):
    tree = ET.parse(file)
    root = tree.getroot()
    is_lsr = root.find('.//metainfo[@type="archive"]/is-lsr')
    if is_lsr is not None:
        is_lsr.text = is_lsr.text.replace('0', '1')
    else:
        archive_metainfo = root.find('.//metainfo[@type="archive"]')
        if archive_metainfo is not None:
            is_lsr = ET.SubElement(archive_metainfo, 'is-lsr')
            is_lsr.text = '1'
    return tree

def get_around_lsr_meta_reindexing(path):
    meta_files = [f for f in os.listdir(path) if os.path.splitext(f)[1] == '.xml']
    for filename in meta_files:
        meta_path = os.path.join(path, filename)
        try:
            with open(meta_path, encoding='utf-8') as file:
                updated_xml = mark_as_lsr(file)

            if updated_xml:
                updated_xml.write(meta_path)
        except (OSError, ET.ParseError):
            logging.debug('Failed to modify metafile:', exc_info=True)


class LSR(object):
    def __init__(self, rpc_uri, account, box):
        self.rpc_uri = rpc_uri
        self.account = account
        self.box = box

    def perform(self, path):
        server, port, cert = self.get_access_to_server()
        logging.info('Using %s:%s for download', server, port)

        self.change_status(104, "Changing LSR status to 'Writing data'... ")

        self.print_starting_information()

        self.run_is_tool(server, port, cert, path)

        get_around_lsr_meta_reindexing(path)

        self.change_status(105, "Changing LSR status to 'Writing data has been completed'... ")
        logging.info(parse_colors('bold_green') + 'The data has been written to the media. Do not forget to deliver the media to the customer.' + parse_colors('reset'))

    def run_is_tool(self, server, port, cert, path):
        try:
            lsr_parameters = ['--lsr', '--server', server, '--port', str(port), '--brand', '1', '--account',
                              self.account, '--box', self.box] + is_lsr.deduce_vault_parameter(path)
            is_lsr.run_is_tool(lsr_parameters, cert)
        except:
            self.change_status(106, "Changing LSR status to 'The order is on hold'... ")
            raise


class AbrLSR(LSR):
    def __init__(self, rpc_uri, hash, insecure):
        _, self.email, self.password_hash, self.box, account = hash.split()
        super().__init__(rpc_uri, account, self.box)
        self.insecure = insecure
        registrations = is_lsr.request_registrations(self.email, self.password_hash, rpc_uri, insecure=insecure)
        lsr_registration = [registration for registration in registrations if suitable_for_lsr(registration, self.box)]
        if not lsr_registration:
            raise NoOrderError.create(registrations, self.box)
        self.order_id = lsr_registration[0]['id']

    def get_access_to_server(self):
        return is_lsr.get_access_to_abr_server(self.rpc_uri, self.email, self.password_hash, self.insecure)

    def change_status(self, status, message):
        return is_lsr.change_status(self.rpc_uri, self.email, self.box, self.password_hash, self.order_id, status, message, insecure=self.insecure)

    def print_starting_information(self):
        logging.info('Writing data for subaccount %s of account %s (%s) by order %s', self.box, self.account, self.email, self.order_id)


class BaaSLSR(LSR):
    def __init__(self, hash, username, password, insecure):
        _, account, machine_id, rpc = hash.split()
        account = account.partition(':')[2]
        self.machine_id = machine_id.partition(':')[2]
        rpc = rpc.partition(':')[2]
        rpc_uri = is_lsr.get_baas_rpc({'rain': rpc})
        box = str(is_lsr.get_machine_subaccount(rpc_uri, account, self.machine_id, username, password, insecure))
        super().__init__(rpc_uri, account, box)
        self.access_parameters = {'email': username, 'password': password, 'insecure': insecure}

    def run_is_tool(self, server, port, cert, path):
        super().run_is_tool(server, port, cert, path)
        backups = find_machine_backups(server, int(port), cert, self.machine_id)
        [self.process_found_backups(backup, server, port, cert, path) for backup in backups if backup]

    def process_found_backups(self, backup, server, port, cert, path):
        lsr_parameters = ['--lsr', '--server', server, '--port', str(port), '--brand', '1', '--account',
                          self.account, '--box', CommonBoxName] + is_lsr.deduce_vault_parameter(path)
        for file in backup:
            lsr_parameters.append('--file')
            lsr_parameters.append(file)
        is_lsr.run_is_tool(lsr_parameters, cert)

    def get_access_to_server(self):
        return is_lsr.get_access_to_baas_server(self.rpc_uri, self.account, **self.access_parameters)

    def change_status(self, status, message):
        pass

    def print_starting_information(self):
        logging.info('Writing data for subaccount %s of account %s', self.box, self.account)


def main(args):
    import sys
    if args.hash is None:
        logging.error("The hash is invalid.")
        sys.exit(1)

    try:
        hash = args.hash.decode('utf-8')

        if hash.startswith('LSR_request'):
            if args.password:
                password = args.password
            else:
                from getpass import getpass
                password = getpass()
            processor = BaaSLSR(hash, args.username, password, args.insecure)
        else:
            processor = AbrLSR(is_lsr.get_abr_rpc(configuration()), hash, args.insecure)
        processor.perform(args.path[0])
    except KeyboardInterrupt:
        logging.error("\nOperation was cancelled.\n")
        sys.exit(1)
    except subprocess.CalledProcessError as e:
        logging.error("An error has occurred.")
        sys.exit(e.returncode)
    except Exception as e:
        logging.error("An error has occurred: " + str(e))
        logging.debug("Exception: ", exc_info=True)
        sys.exit(1)

def parse_arguments():
    import argparse
    parser = argparse.ArgumentParser(add_help=True, description='Tool to download Large Scale Recovery backups from Acronis storage.')
    parser.add_argument('--username', required=False, help="service account that performs LSR")
    parser.add_argument('--password', required=False, help="service account credentials to perform LSR")
    parser.add_argument('--secure', dest='insecure', action='store_false', help='forces SSL certificate validation')
    parser.add_argument('--insecure', dest='insecure', action='store_true', help='allows connection to a server with a not trusted SSL certificate')
    parser.add_argument('hash', metavar='<hash>', type=is_lsr.Base64Type(), help='a hash code that must be obtained from the Large Scale Recovery management panel')
    parser.add_argument('path', metavar='<path>', nargs='+', help='one or more paths where the backups will be copied')
    parser.set_defaults(insecure=False)
    return parser.parse_args()

if __name__ == "__main__":
    colorama_init()
    from pretty_logging import setup_logging
    setup_logging()
    main(parse_arguments())
