#!/usr/bin/env python
# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

import is_lsr
import logging
import subprocess
import xml.etree.ElementTree as ET
from colorama import init as colorama_init
from colorlog.escape_codes import parse_colors
from is_lsr import configuration, is_tool_path
from rpc_client import execute_rpc_request


def read_meta(vault_parameter):
    result = subprocess.check_output([is_tool_path(), '--parameters'] + vault_parameter)
    return ET.fromstring(result)

class IS(object):
    def __init__(self, rpc_uri, account, box, insecure):
        self.rpc_uri = rpc_uri
        self.account = account
        self.box = box
        self.insecure = insecure

    def perform(self, path):
        server, port, cert = self.get_access_to_server()
        logging.info('Using %s:%s for upload', server, port)

        self.change_status(17, "Changing IS status to 'The data upload has started'... ")

        self.print_starting_information()

        try:
            is_parameters = ['--is', '--server', server, '--port', str(port), '--brand', '1', '--account', self.account, '--box', self.box] + is_lsr.deduce_vault_parameter(path)
            is_lsr.run_is_tool(is_parameters, cert)
        except:
            self.change_status(19, "Changing IS status to 'The order is on hold'... ")
            raise

        self.change_status(18, "Changing IS status to 'The data upload has been completed'... ")
        self.update_used_space()
        logging.info(parse_colors('bold_green') + 'The Initial Seeding backup has been uploaded to the data center.' + parse_colors('reset'))


class AbrIS(IS):
    def __init__(self, rpc_uri, hash, account, box, session, insecure):
        hash_brand, self.email, self.password_hash, hash_box, hash_account = hash.decode().splitlines()
        super().__init__(rpc_uri, hash_account, box, insecure)
        if self.email != account:
            logging.warning('There is a discrepancy in the user email taken from the archive and from the order.')
        if self.box != hash_box:
            logging.warning('There is a discrepancy in the subaccount taken from the archive and from the order.' +
                            'The archive may be inaccessible.')
        logging.debug('Requesting available licenses from the RPC')
        registrations = is_lsr.request_registrations(self.email, self.password_hash, self.rpc_uri, self.insecure)
        is_registration = [registration for registration in registrations
                           if registration['ProductType'] == 'IS' and registration['SessionId'] == session]
        if not is_registration:
            raise Exception('No Initial Seeding orders are ready for processing.')
        self.order_id = is_registration[0]['id']

    def get_access_to_server(self):
        return is_lsr.get_access_to_abr_server(self.rpc_uri, self.email, self.password_hash, insecure=self.insecure)

    def change_status(self, status, message):
        return is_lsr.change_status(self.rpc_uri, self.email, self.box, self.password_hash, self.order_id, status, message, insecure=self.insecure)

    def print_starting_information(self):
        logging.info('Uploading archives for subaccount %s of account %s (%s) by order %s', self.box, self.account, self.email, self.order_id)

    def update_used_space(self):
        execute_rpc_request(self.rpc_uri, 'UpdateStorageSize', email=self.email, password_hash=self.password_hash, subaccountId=self.box, insecure=self.insecure)


class BaaSIS(IS):
    def __init__(self, rpc_uri, account, box, token, insecure):
        super().__init__(rpc_uri, account, box, insecure)
        self.token = token

    def get_access_to_server(self):
        return is_lsr.get_access_to_baas_server(self.rpc_uri, None, self.token, email=self.account, insecure=self.insecure)

    def change_status(self, status, msg):
        pass

    def print_starting_information(self):
        logging.info('Uploading archives for subaccount %s of account %s', self.box, self.account)

    def update_used_space(self):
        pass


def main(args):
    import sys
    try:
        vault_parameter = is_lsr.deduce_vault_parameter(args.path[0])
        meta = read_meta(vault_parameter)

        account = meta.get('account_id', '')
        box = meta.get('subaccount_id', '')
        session = meta.get('session_id', None)
        token = meta.get('token', None)

        if hasattr(args, 'hash'):
            if session is None:
                if token is not None:
                    raise Exception('Wrong usage: this type of IS archives does not require session id.')
                else:
                    raise Exception('Invalid IS archive. The session id is missing.')
            processor = AbrIS(is_lsr.get_abr_rpc(configuration()), args.hash, account, box, session, args.insecure)
        else:
            if token is None:
                if session is not None:
                    raise Exception('You forgot to specify session hash. The session id is {0}'.format(session))
                else:
                    raise Exception('Invalid IS archive. The token is missing.')
            processor = BaaSIS(is_lsr.get_baas_rpc(configuration()), account, box, token, args.insecure)
        processor.perform(args.path[0])
    except KeyboardInterrupt:
        logging.error("\nOperation was cancelled.")
        sys.exit(1)
    except subprocess.CalledProcessError as e:
        logging.error("An error has occurred.")
        sys.exit(e.returncode)
    except Exception as e:
        logging.error("An error has occurred: " + str(e))
        logging.debug("Exception: ", exc_info=True)
        sys.exit(1)

def parse_arguments():
    import argparse
    parser = argparse.ArgumentParser(add_help=True, description='Tool to upload Initial Seeding backup to Acronis storage.')
    parser.add_argument('hash', metavar='<hash>', type=is_lsr.Base64Type(), nargs='?', default=argparse.SUPPRESS,
                        help='a hash code that must be obtained from the Initial Seeding management panel')
    parser.add_argument('path', metavar='<path>', nargs='+', help='path to an IS backup or the .TIB file')
    parser.add_argument('--secure', dest='insecure', action='store_false', help='forces SSL certificate validation')
    parser.add_argument('--insecure', dest='insecure', action='store_true', help='allows connection to a server with a not trusted SSL certificate')
    parser.set_defaults(insecure=False)
    return parser.parse_args()

if __name__ == "__main__":
    colorama_init()
    from pretty_logging import setup_logging
    setup_logging()
    main(parse_arguments())
