# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

import base64
import get_certificate
import logging
import os
import rpc_response
import subprocess
from argparse import SUPPRESS
from contextlib import contextmanager
from rpc_client import execute_rpc_request
from urllib.parse import urljoin


class Base64Type(object):
    def __call__(self, string):
        try:
            return base64.b64decode(string, validate=True)
        except base64.binascii.Error:
            return SUPPRESS


class Error(Exception):
    pass


class CertificateError(Error):
    def __init__(self, ex):
        self.exception = ex

    def __str__(self):
        return 'Failed to load the certificate: ' + str(self.exception)


class RedirectError(Error):
    def __init__(self, ex):
        self.exception = ex

    def __str__(self):
        return 'Failed to resolve the storage: ' + str(self.exception)


class ConfigurationError(Error):
    def __init__(self, ex):
        self.exception = ex

    def __str__(self):
        return 'Failed to read the configuration file: ' + str(self.exception)

def is_tool_path():
    return os.path.join(os.path.dirname(os.path.abspath(__file__)), 'is_tool.exe')

def run_is_tool(args, input):
    logging.debug('Executing is_tool process with parameters %s', str(args))
    process = subprocess.Popen([is_tool_path()] + args, stdin=subprocess.PIPE)
    _, error = process.communicate(input=input)
    if process.returncode:
        exception = subprocess.CalledProcessError(process.returncode, process.args, output=error)
        logging.debug(exception)
        raise exception

def configuration():
    configuration_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'configuration')
    from parse_config import parse
    return parse(configuration_path)

def deduce_vault_parameter(path):
    if os.path.isdir(path):
        vault_parameter = ['--vault', path]
    elif os.path.isfile(path):
        vault_parameter = ['--file', path]
    else:
        raise FileNotFoundError('File or directory {0} is not found.'.format(path))
    return vault_parameter

def get_abr_rpc(configuration):
    try:
        return urljoin(configuration['rpc'], '/online-backup/2.0/')
    except Exception as ex:
        raise ConfigurationError(ex) from ex

def get_baas_rpc(configuration):
    try:
        return urljoin(configuration['rain'], '/api/1/rpc/')
    except Exception as ex:
        raise ConfigurationError(ex) from ex

def get_handler(logger, name, default=None):
    handlers = [handler for handler in logger.handlers if handler.name == name]
    return default if len(handlers) == 0 else handlers[0]

@contextmanager
def change_terminator(handler, new_terminator=''):
    original_terminator = handler.terminator
    handler.terminator = new_terminator
    try:
        yield
    finally:
        handler.terminator = original_terminator

def get_machine_subaccount(rpc_uri, account, machine_id, username, password, insecure=False):
    logging.debug('Requesting list of subaccounts from the RAIN')
    parameters = {
        'email': username,
        'password': password,
        'account': account,
        'machine': machine_id,
        'service': True,
        'insecure': insecure,
        'response_decoder': rpc_response.decode_getSubaccountsList_response
    }
    subaccount = execute_rpc_request(rpc_uri, 'GetSubaccountsList', **parameters)[0]
    logging.debug('Resulting subaccount: %s', subaccount)
    return subaccount

def get_access_to_baas_server(rpc_uri, account, token=None, email=None, password=None, insecure=False):
    authority = get_certificate.CertificateAuthority(rpc_uri, email, password, token, writable=True, service=False, insecure=insecure)
    try:
        logging.debug('Requesting storage parameters from the RPC/RegServer')
        server, port = authority.get_storage(account).split(':')
    except Exception as e:
        raise RedirectError(e) from e

    try:
        logging.debug('Issuing a certificate')
        cert = authority.issue(account, None)
    except Exception as e:
        raise CertificateError(e) from e

    return server, port, cert

def get_access_to_abr_server(rpc_uri, account, password_hash, insecure=False):
    try:
        logging.debug('Requesting RegServer address from the RPC')
        parameters = {
            'email': account,
            'password_hash': password_hash,
            'insecure': insecure,
            'response_decoder': rpc_response.decode_redirect_response
        }
        response = execute_rpc_request(rpc_uri, 'GetRedirect', **parameters)

        reg_server, cipher = (response['Address'], response['Query'])
        if cipher.startswith('cipher='):
            cipher = base64.b64decode(cipher[7:])

        logging.debug('Requesting storage address from the RegServer')
        authority = get_certificate.CertificateAuthority(rpc_uri, None, password_hash, None, insecure=True)
        reg_server = reg_server.replace('http', 'https').replace('httpss', 'https')
        server, port = authority.execute_request('GetDatacenterAddress', reg_server, cipher, context=authority._ctx).split(':')
    except Exception as e:
        raise RedirectError(e) from e

    try:
        logging.debug('Reading the certificate')
        with open(os.path.join(os.path.dirname(__file__), 'web-client.pem')) as certificate_file:
            cert = certificate_file.read().encode('utf8')
    except Exception as e:
        raise CertificateError(e) from e

    return server, port, cert

def request_registrations(email, hash_pass, rpc_uri, insecure=False):
    parameters = {
        'email': email,
        'password_hash': hash_pass,
        'insecure': insecure,
        'response_decoder': rpc_response.decode_registrations_response
    }
    return execute_rpc_request(rpc_uri, 'GetRegistrations', **parameters)

def change_status(rpc_uri, account, box, password_hash, order_id, status, message, insecure=False):
    logger = logging.getLogger()
    console_handler = get_handler(logger, 'console', logger.handlers[0])
    with change_terminator(console_handler):
        logging.info(message)
        parameters = {
            'email': account,
            'password_hash': password_hash,
            'subaccountId': box,
            'orderId': order_id,
            'status': status,
            'insecure': insecure
        }
        execute_rpc_request(rpc_uri, 'UpdateSubaccountRegistrationStatus', **parameters)
    logging.info('OK')
