#!/usr/bin/env python
# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

from acronis import storage
from base64 import decodebytes
from OpenSSL import crypto
from rpc_client import execute_rpc_request
from rpc_response import decode_redirect_response
from ssl import create_default_context, CERT_NONE
import logging
import sys

_RSA_KEY_LENGTH = 2048


class CertificateAuthority(storage.CertificateAuthority):
    def __init__(self, host, login, password=None, token=None, writable=False, service=True, insecure=False):
        super().__init__()

        self._host = host
        self._login = login
        self._password = password
        self._token = token
        self._writable = writable
        self._service = service
        self._ctx = create_default_context()
        if insecure:
            self._ctx.check_hostname = False
            self._ctx.verify_mode = CERT_NONE

    def get_service_access(self, account_id, subaccount_id=None):
        parameters = {
            'email': self._login,
            'password': self._password,
            'token': self._token,
            'account': account_id,
            'subaccountId': subaccount_id
        }
        if self._service:
            parameters['service'] = 'downloader'
        response = execute_rpc_request(self._host, 'GetRedirect', response_decoder=decode_redirect_response, context=self._ctx, **parameters)

        return ':'.join((response['Host'], response['Port'])),\
               decodebytes(response['Cipher'].encode('ascii')), response['AccountId'], response['BrandId']

    def get_redirect(self, account_id, subaccount_id=None):
        host, cipher, _, _ = self.get_service_access(account_id, subaccount_id=subaccount_id)
        return host, cipher

    def issue(self, account_id, subaccount_id=None):
        key = crypto.PKey()
        key.generate_key(crypto.TYPE_RSA, 2048)
        request = crypto.X509Req()
        request.set_pubkey(key)
        request.set_version(3)
        request.sign(key, 'sha256')
        encrypted_request = crypto.dump_certificate_request(crypto.FILETYPE_ASN1, request)
        requestType = 'CreateAccountCertificate' if self._writable else 'CreateTempCertificate'
        certificate = self.request(requestType, account_id, encrypted_request, context=self._ctx).data

        return b''.join((crypto.dump_certificate(crypto.FILETYPE_PEM, crypto.load_certificate(crypto.FILETYPE_ASN1, certificate)),
                         crypto.dump_privatekey(crypto.FILETYPE_PEM, key)))

    def get_storage(self, account_id):
        """
        :param account_id: unique identifier of account
        :return: storage_address
        """
        return self.request('GetDatacenterAddress', account_id, context=self._ctx)


def main(args):
    try:
        logging.basicConfig(stream=sys.stderr, level=logging.DEBUG)
        authority = CertificateAuthority(args.rpc, args.username, args.password, args.token, args.writable, not args.client, args.insecure)
        print(authority.issue(args.account).decode('utf8'))
    except Exception as e:
        import traceback
        print('Failed: %r' % e)
        traceback.print_exc()

def parse_arguments():
    from argparse import ArgumentParser

    parser = ArgumentParser(add_help=False)
    parser.add_argument('--writable', action='store_true', help='request a writable certificate')
    parser.add_argument('--client', action='store_true', help='request not a service certificate')
    parser.add_argument('--username', help='account name')
    parser.add_argument('--password', help='account password')
    parser.add_argument('--account', help='account id')
    parser.add_argument('--rpc', help='rpc address', required=True)
    parser.add_argument('--token', help='security token')
    parser.add_argument('--secure', dest='insecure', action='store_false', help='Forces SSL certificate validation')
    parser.add_argument('--insecure', dest='insecure', action='store_true', help='Allows connection to a server with not trusted SSL certificate')
    parser.set_defaults(insecure=False)

    return parser.parse_args()

if __name__ == "__main__":
    main(parse_arguments())
