# @copyright (c) 2002-2016 Acronis International GmbH. All rights reserved.
# EULA: https://www.acronis.com/en-us/download/docs/eula/corporate/

import asyncio
import logging
import os.path
import xml.etree.ElementTree as ET
from fes.client import create_client
from tempfile import TemporaryDirectory

CommonBoxName = '1'

def deduce_meta_jrnl(tib_name, files):
    if tib_name + '.jrnl' in files:
        return tib_name + '.jrnl'
    elif tib_name + '.meta' in files:
        return tib_name + '.meta'
    else:
        return None

def is_meta_contains_id(meta, tag, id):
    if meta is None:
        return False
    meta_id = meta.find(tag).text
    return meta_id and meta_id.strip() == id

def is_archive_of_machine(archive_meta, machine_id):
    return is_meta_contains_id(archive_meta, 'machine-id', machine_id)

def is_raw_archive_of_archive(raw_archive, archive_key):
    return is_meta_contains_id(raw_archive, 'archive-key', archive_key)

def parse_meta(meta_data, machine_id, xml_name, files):
    logging.debug('Parsing %s', xml_name)
    try:
        meta = ET.fromstring(meta_data)
        archive_meta = meta.find("metainfo[@type='archive']")
        if is_archive_of_machine(archive_meta, machine_id):
            logging.debug('Archive meta: %s', archive_meta.attrib)
            archive_key = archive_meta.attrib['id']
            raw_archive = meta.find("metainfo[@type='raw-archive']")
            if is_raw_archive_of_archive(raw_archive, archive_key):
                logging.debug('Raw archive meta: %s', raw_archive.attrib)
                tib_name = raw_archive.find('name').text
                tib_name = tib_name.strip() if tib_name else None
                meta_or_jrnl_name = deduce_meta_jrnl(tib_name, files)
                return (xml_name, tib_name, meta_or_jrnl_name) if meta_or_jrnl_name else None
    except (ET.ParseError, KeyError):
        logging.debug('Failed to parse metafile %s:', xml_name, exc_info=True)

async def load_meta(xml_name, machine_id, client, files):
    logging.debug('Found XML: %s', xml_name)
    with await client.open_input_stream(xml_name, CommonBoxName) as stream:
        meta_data = await stream.read()
        return parse_meta(meta_data, machine_id, xml_name, files)

async def list_directory_meta(loop, server, port, certificate, machine_id):
    with TemporaryDirectory() as tmp_dir_name:
        tmp_file_name = os.path.join(tmp_dir_name, 'tmp.crt')
        with open(tmp_file_name, mode='w+b') as cert_file:
            cert_file.write(certificate)
        with await create_client(server, port, tmp_file_name, loop) as client:
            files = await client.get_file_info_list(CommonBoxName + '::')
            files = [item['name'] for item in files['items']]
            tasks = [load_meta(file, machine_id, client, files) for file in files if file.endswith('.xml')]
            return await asyncio.gather(*tasks, loop=loop)

def find_machine_backups(server, port, certificate, machine_id):
    loop = asyncio.get_event_loop()
    try:
        task = list_directory_meta(loop, server, port, certificate, machine_id)
        return loop.run_until_complete(task)
    finally:
        loop.close()
